<?php
namespace Kanews\Util;

class Kai {

  private static $instance;

  public static function getInstance() {
    if ( null === static::$instance ) {
      static::$instance = new static();
    }
    return static::$instance;
  }

  public function __construct() {
    add_action('add_meta_boxes',        [ $this,'add_meta_box'] );
    add_action('admin_footer',          [ $this,'popup_content'] );
    add_action('admin_enqueue_scripts', [ $this,'enqueue_scripts'] );
    add_action('wp_footer',             [ $this,'add_chat_to_content'] );
    add_action('wp_ajax_kanews_chat_message',        [ $this, 'handle_chat_message' ]);
    add_action('wp_ajax_nopriv_kanews_chat_message', [ $this, 'handle_chat_message' ]);
  }

  public static function get_api_key($api) {
    if( $api == 'openai' ) {
      return kanews_get_option('openai-api', '');
    } else {
      return kanews_get_option('pexels-api', '');
    }
  }

  public static function get_site_credits() {
    return get_option('kthms_ai_credits', array('credits' => 100));
  }

  public static function reduce_site_credits($amount) {
    if( ! empty(Kai::get_api_key('openai')) )
        return true;

    $current_credits = self::get_site_credits();
    
    if ($current_credits['credits'] >= $amount) {
        $new_credits = array('credits' => $current_credits['credits'] - $amount);
        update_option('kthms_ai_credits', $new_credits);
        return true;
    } else {
        return false;
    }
  }

  function add_meta_box() {
    add_meta_box(
        'kanews_ai_meta_box',
        'Kanews AI <span class="kanews-ai-badge" style="font-size: 11px; font-weight: 500; padding: 4px 10px; border-radius: 3px; margin-left: 5px; color: #10AC84; background: rgba(16,172,132,0.1);">KAI</span>',
        [ $this, 'kanews_ai_meta_box_callback' ],
        array('post', 'columist'),
        'side',
        'high'
    );
  }

  public static function kanews_ai_meta_box_callback() {
    echo '<div class="kanews-ai-metabox-trigger"><br>
    <div class="components-flex components-h-stack components-v-stack css-8mn8b1 e19lxcc00">
        <div id="kanews-ai-launch" class="components-button button editor-post-trash is-next-40px-default-size is-secondary" style="display:flex;align-items:center"><svg width="24" height="24" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M3 19V5.70046C3 5.27995 3.26307 4.90437 3.65826 4.76067L13.3291 1.24398C13.5886 1.14961 13.8755 1.28349 13.9699 1.54301C13.9898 1.59778 14 1.65561 14 1.71388V6.6667L20.3162 8.77211C20.7246 8.90822 21 9.29036 21 9.72079V19H23V21H1V19H3ZM5 19H12V3.85543L5 6.40089V19ZM19 19V10.4416L14 8.77488V19H19Z"></path></svg> KAI\'yi Başlat</div>
    </div></div>';
    wp_nonce_field('kanews_ai_nonce_action', 'kanews_ai_nonce_name');
  }

  function popup_content() {
    $screen = get_current_screen();
    if ($screen->base !== 'post' && $screen->base !== 'post-new') return;
    
    // Check API key statuses
    $openai_status = !empty(Kai::get_api_key('openai'));
    $pexels_status = !empty(Kai::get_api_key('pexels'));
    
    ?>
    <div id="kanews-ai-popup" class="kanews-ai-popup" style="display:none">
        <div class="kanews-ai-popup-content">
            <div class="kanews-ai-popup-header">
                <h2>Kanews AI <span class="kanews-ai-badge">KAI</span></h2>
                <span class="kanews-ai-popup-close">&times;</span>
            </div>
            
            <div class="kanews-ai-popup-body">
                <?php if (!$openai_status || !$pexels_status): // Check if any API key is inactive ?>
                    <div class="kanews-ai-warning" style="color: red; margin-bottom: 10px;">
                        <strong>Uyarı:</strong> API Key'ler pasif. Lütfen ayarları kontrol edin.
                    </div>
                <?php endif; ?>
                
                <div class="kanews-ai-task-selector">
                    <select id="kanews-ai-selector" class="components-button editor-post-trash is-next-40px-default-size is-secondary">
                        <option value="">Görev Seç</option>
                        <option value="tts">Ses Dosyası Oluştur</option>
                        <option value="excerpt">Yazı Açıklaması</option>
                        <option value="summary">Yazı Özeti</option>
                        <option value="seo-title">SEO Başlığı</option>
                        <option value="article">Makale Oluştur (BETA)</option>
                        <option value="paraphrase">Makale Özgünleştirici</option>
                        <option value="featured-image">Öne Çıkarılan Görsel</option>
                        <option value="comments">Yorum Oluştur</option>
                        <option value="faq">SSS Oluştur</option>
                    </select>
                    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" width="24" height="24"><path d="M11.9999 13.1714L16.9497 8.22168L18.3639 9.63589L11.9999 15.9999L5.63599 9.63589L7.0502 8.22168L11.9999 13.1714Z"></path></svg>
                </div>
                
                <!-- Task specific input areas -->
                <div id="kanews-ai-task-inputs"></div>
                
                <!-- Results area -->
                <div id="kanews-ai-results" class="kanews-ai-results"></div>
            </div>
            
            <div class="kanews-ai-popup-footer">
                <div class="kanews-ai-api-status">
                    <div class="api-status-item">
                        <span class="label">OperAI API:</span>
                        <span class="status <?php echo $openai_status ? 'active' : 'inactive'; ?>">
                            <?php echo $openai_status ? 'Aktif' : 'Pasif'; ?>
                        </span>
                    </div>
                    <div class="api-status-item">
                        <span class="label">Pexels API:</span>
                        <span class="status <?php echo $pexels_status ? 'active' : 'inactive'; ?>">
                            <?php echo $pexels_status ? 'Aktif' : 'Pasif'; ?>
                        </span>
                    </div>
                    <div class="api-status-item">
                        <span class="label">v2.1.1</span>
                    </div>
                </div>
                <div style="display: flex; align-items: center; justify-content: space-between;gap:10px">
                    <div id="kanews-ai-loading"></div>
                    <button id="kanews-ai-execute" class="components-button is-primary button" disabled>Görevi Başlat</button>
                </div>
            </div>
        </div>
    </div>
    <?php
  }

  function enqueue_scripts($hook) {
    if ('post.php' !== $hook && 'post-new.php' !== $hook) {
        return;
    }
    
    wp_enqueue_style('kanews-ai-admin-style', KAN_TEMPLATE_URL . '/assets/css/admin.css', array(), null);
    wp_enqueue_script('kanews-ai-script', KAN_TEMPLATE_URL . '/assets/js/admin/kanews-ai.js', array('jquery'), null, true);
  }

  public function add_chat_to_content() { 
    // Only add chat to single posts
    if (!is_single()) {
      return;
    }

    // Check global theme setting
    $chat_active = kanews_get_option('kai-chat-active', false);
    
    // Check post-specific override
    $post_override = get_post_meta(get_the_ID(), 'kai-chat-override', true);
    if ($post_override === 'on') {
        $chat_active = true;
    } elseif ($post_override === 'off') {
        $chat_active = false;
    }

    if (!$chat_active) {
        return;
    }

    // Check if API key is configured
    $api_key = self::get_api_key('openai');
    if (empty($api_key)) {
      echo '<div class="kanews-chat">
        <div class="kanews-chat-button">
          <i class="icon-comment"></i>
        </div>
        <div class="kanews-chat-window">
          <div class="kanews-chat-header">
            <div class="kanews-chat-title">' . esc_html(kanews_get_option('kai-chat-title', 'KAI ile Haber Hakkında Sohbet')) . '</div>
            <div class="close-chat"><i class="icon-close"></i></div>
          </div>
          <div class="kanews-chat-messages">
            <div class="kanews-chat-message system-message">
              Sohbet sistemi şu anda aktif değil. Lütfen daha sonra tekrar deneyin.
            </div>
          </div>
        </div>
      </div>';
      return;
    }

    

    // Get customizable options from theme settings
    $chat_title = kanews_get_option('kai-chat-title', 'KAI ile Haber Hakkında Sohbet');
    $input_placeholder = kanews_get_option('kai-chat-placeholder', 'Sorunuzu yazın...');
  

    $chat_html = '
    <div class="kanews-chat">
      <div class="kanews-chat-button" role="tooltip" data-microtip-position="left" aria-label="'.esc_attr($chat_title).'">
        <i class="icon-comment"></i>
      </div>
      <div class="kanews-chat-window">
        <div class="kanews-chat-header">
          <div class="kanews-chat-title">' . esc_html($chat_title) . '</div>
          <div class="close-chat"><i class="icon-close"></i></div>
        </div>
        <div class="kanews-chat-messages"></div>
        <div class="kanews-chat-remaining"></div>
        <div class="kanews-chat-input">
          <input type="text" placeholder="' . esc_attr($input_placeholder) . '">
          <button type="button">
            <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
              <line x1="22" y1="2" x2="11" y2="13"></line>
              <polygon points="22 2 15 22 11 13 2 9 22 2"></polygon>
            </svg>
          </button>
          <input type="hidden" name="kanews_chat_nonce" id="kanews_chat_nonce" value="' . esc_attr($nonce) . '">
        </div>
      </div>
    </div>';

    echo $chat_html;
  }

  public function handle_chat_message() {
    // Verify nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'kanews_chat_nonce')) {
      wp_send_json_error(['message' => 'Güvenlik doğrulaması başarısız.']);
      return;
    }

    // Check if chat is enabled for this post
    $chat_active = kanews_get_option('kai-chat-active', true);
    $post_override = get_post_meta(intval($_POST['article_id']), 'kai-chat-override', true);
    
    if ($post_override === 'on') {
        $chat_active = true;
    } elseif ($post_override === 'off') {
        $chat_active = false;
    }

    if (!$chat_active) {
      wp_send_json_error(['message' => 'Sohbet sistemi bu makale için devre dışı.']);
      return;
    }

    // Check subscription status
    $is_subscribed = kanews_get_option('kanews-insights', false);
    $subscription_url = kanews_insights_page_url();

    // Get message count from cookie
    $cookie_name = 'kanews_chat_count_' . intval($_POST['article_id']);
    $message_count = isset($_COOKIE[$cookie_name]) ? intval($_COOKIE[$cookie_name]) : 0;

    // Get limits from theme settings
    $free_limit = kanews_get_option('kai-chat-free-limit', 5);
    $premium_limit = kanews_get_option('kai-chat-premium-limit', 20);
    
    // Check message limits
    $limit = $is_subscribed ? $premium_limit : $free_limit;
    if ($message_count >= $limit) {
      $error_message = $is_subscribed 
        ? 'Bu makale için soru limitinize ulaştınız.'
        : sprintf('Ücretsiz soru limitinize ulaştınız. Daha fazla soru sormak için <a href="%s">premium üyelik</a> satın alın.', esc_url($subscription_url));
      wp_send_json_error(['message' => $error_message]);
      return;
    }

    // Get and validate input
    $message = sanitize_text_field($_POST['message'] ?? '');
    $article_id = intval($_POST['article_id'] ?? 0);
    $article_title = sanitize_text_field($_POST['article_title'] ?? '');
    $article_content = sanitize_text_field($_POST['article_content'] ?? '');

    if (empty($message) || empty($article_id)) {
      wp_send_json_error(['message' => 'Geçersiz istek parametreleri.']);
      return;
    }

    try {
      // Get OpenAI API key
      $api_key = self::get_api_key('openai');
      if (empty($api_key)) {
        wp_send_json_error(['message' => 'OpenAI API anahtarı bulunamadı.']);
        return;
      }

      // Prepare system message with article context
      $system_message = "Sen bir haber makalesi hakkında sohbet eden yardımcı bir asistansın. " .
                       "Makale başlığı: {$article_title}\n" .
                       "Makale içeriği özeti: " . substr($article_content, 0, 500) . "...\n" .
                       "Makale hakkında sorulan sorulara kısa ve öz yanıtlar ver.";

      // Prepare the API request
      $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
          'Content-Type' => 'application/json',
        ],
        'body' => json_encode([
          'model' => 'gpt-3.5-turbo',
          'messages' => [
            ['role' => 'system', 'content' => $system_message],
            ['role' => 'user', 'content' => $message]
          ],
          'max_tokens' => 150,
          'temperature' => 0.7,
        ]),
        'timeout' => 30,
      ]);

      if (is_wp_error($response)) {
        throw new Exception($response->get_error_message());
      }

      $body = json_decode(wp_remote_retrieve_body($response), true);
      
      if (isset($body['error'])) {
        throw new Exception($body['error']['message']);
      }

      if (!isset($body['choices'][0]['message']['content'])) {
        throw new Exception('Beklenmeyen API yanıtı.');
      }

      $ai_response = trim($body['choices'][0]['message']['content']);

      // Increment message count cookie
      $new_count = $message_count + 1;
      setcookie($cookie_name, $new_count, time() + 86400, '/'); // Cookie expires in 24 hours

      // Add remaining messages info to response
      $remaining = $limit - $new_count;
      $status_message = $is_subscribed 
        ? sprintf('Premium üye olarak %d mesaj hakkınız kaldı.', $remaining)
        : sprintf('Ücretsiz kullanıcı olarak %d mesaj hakkınız kaldı. Daha fazlası için <a href="%s">premium üyelik</a> satın alın.', $remaining, esc_url($subscription_url));

      wp_send_json_success([
        'message' => $ai_response,
        'remaining' => $remaining,
        'status_message' => $status_message
      ]);

    } catch (Exception $e) {
      wp_send_json_error(['message' => 'API Hatası: ' . $e->getMessage()]);
    }
  }

}