<?php
/**
 * Dynamic CSS Generator for Gutenberg Blocks
 *
 * @package Kanews
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Dynamic CSS Generator Class
 */
class Kanews_Dynamic_CSS {

    /**
     * Instance of this class.
     *
     * @var object
     */
    protected static $instance = null;

    /**
     * CSS rules array
     *
     * @var array
     */
    protected $css_rules = [];

    /**
     * Get instance of this class.
     *
     * @return Kanews_Dynamic_CSS
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor.
     */
    public function __construct() {
        // add_action('wp_head', [$this, 'output_css'], 100);
        // add_action('admin_head', [$this, 'output_css'], 100);
        // add_action('wp_enqueue_scripts', [$this, 'enqueue_dynamic_css'], 100);
        // add_action('admin_enqueue_scripts', [$this, 'enqueue_dynamic_css'], 100);
    }

    /**
     * Add CSS rule
     *
     * @param string $selector CSS selector
     * @param array  $rules    CSS rules
     * @param string $media    Media query
     */
    public function add_rule($selector, $rules, $media = '') {
        if (!isset($this->css_rules[$media])) {
            $this->css_rules[$media] = [];
        }

        if (!isset($this->css_rules[$media][$selector])) {
            $this->css_rules[$media][$selector] = [];
        }

        $this->css_rules[$media][$selector] = array_merge($this->css_rules[$media][$selector], $rules);
    }

    /**
     * Add typography CSS
     *
     * @param string $selector CSS selector
     * @param array  $typography Typography settings
     */
    public function add_typography($selector, $typography) {
        $rules = [];

        if (!empty($typography)) {
            // Font Family
            if (!empty($typography['fontFamily'])) {
                $rules['font-family'] = $typography['fontFamily'];
            }

            // Font Size
            if (!empty($typography['fontSize'])) {
                $rules['font-size'] = $typography['fontSize'];
            }

            // Font Weight
            if (!empty($typography['fontWeight'])) {
                $rules['font-weight'] = $typography['fontWeight'];
            }

            // Line Height
            if (!empty($typography['lineHeight'])) {
                $rules['line-height'] = $typography['lineHeight'];
            }

            // Letter Spacing
            if (!empty($typography['letterSpacing'])) {
                $rules['letter-spacing'] = $typography['letterSpacing'];
            }

            // Text Transform
            if (!empty($typography['textTransform']) && $typography['textTransform'] !== 'none') {
                $rules['text-transform'] = $typography['textTransform'];
            }

            // Font Style
            if (!empty($typography['fontStyle']) && $typography['fontStyle'] !== 'normal') {
                $rules['font-style'] = $typography['fontStyle'];
            }

            // Text Decoration
            if (!empty($typography['textDecoration']) && $typography['textDecoration'] !== 'none') {
                $rules['text-decoration'] = $typography['textDecoration'];
            }

            // Text Align
            if (!empty($typography['textAlign']) && $typography['textAlign'] !== 'left') {
                $rules['text-align'] = $typography['textAlign'];
            }

            // Color
            if (!empty($typography['color'])) {
                $rules['color'] = $typography['color'];
            }
        }

        if (!empty($rules)) {
            $this->add_rule($selector, $rules);
        }
    }

    /**
     * Generate CSS from rules
     *
     * @return string
     */
    protected function generate_css() {
        $css = '';

        foreach ($this->css_rules as $media => $selectors) {
            if (!empty($media)) {
                $css .= "@media {$media} {\n";
            }

            foreach ($selectors as $selector => $rules) {
                $css .= "{$selector} {\n";
                foreach ($rules as $property => $value) {
                    $css .= "    {$property}: {$value};\n";
                }
                $css .= "}\n";
            }

            if (!empty($media)) {
                $css .= "}\n";
            }
        }

        return $css;
    }

    /**
     * Output CSS to head
     */
    public function output_css() {
        $css = $this->generate_css();
        if (!empty($css)) {
            echo "<style id='kanews-dynamic-css'>\n{$css}\n</style>\n";
        }
    }

    /**
     * Enqueue dynamic CSS
     */
    public function enqueue_dynamic_css() {
        $css = $this->generate_css();
        if (!empty($css)) {
            wp_register_style('kanews-dynamic-css', false);
            wp_enqueue_style('kanews-dynamic-css');
            wp_add_inline_style('kanews-dynamic-css', $css);
        }
    }
}

// Initialize the class
Kanews_Dynamic_CSS::get_instance(); 