<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use controlElementor\Widgets\Traits\Text_Trait;

/**
 * Elementor feature widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class About_Company extends Widget_Base {

	use Text_Trait;

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'about_company';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'About Company', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-image-before-after';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'content', 'image gallery' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'About Company', 'control-elementor' ),
			]
		);

		$this->add_control(
			'gallery',
			[
				'label' => esc_html__( 'Image gallery', 'elementor' ),
				'type' => Controls_Manager::GALLERY,
				'default' => []
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Heading title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_html__( 'About Company', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'content',
			[
				'label' => esc_html__( 'Description', 'control-elementor' ),
				'type' => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => '<h4>Lead text Investment Expert is one of the largest and most experienced global private equity firms. We have established a team investment professionals,foucsed on buyouts and </h4>
				<p class="normal">Many businesses,large and small,have a huge source of great ideas that can help theme improve,innovate and grow,and yet so many of these companies never think of using this amazing corporate asset. What is this highly valuable asset? Its own people.Says Morgan Fraud,the author of The Investment Experts."Given that we are all capable of contributing new ideas,the"</p>',
			]
		);

		$this->add_control(
			'footer_text',
			[
				'label' => esc_html__( 'Footer text', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		//gallery settings
		$this->start_controls_section(
			'image-settings',
			[
				'label' => esc_html__( 'Image Settings', 'control-elementor' ),
			]
		);

		$this->add_control(
			'large_img_size',
			[
				'label' => esc_html__( 'First image size', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => control_elementor_get_image_sizes_Arr(),
				'default' => 'investment-630x335-crop',
			]
		);

		$this->add_control(
			'thumb_img_size',
			[
				'label' => esc_html__( 'Small image size', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => control_elementor_get_image_sizes_Arr(),
				'default' => 'investment-300x300-crop',
			]
		);

		$this->add_control(
			'position',
			[
				'label' => esc_html__( 'Image Position', 'elementor' ),
				'type' => Controls_Manager::CHOOSE,
				'default' => '',
				'options' => [
					'no' => [
						'title' => esc_html__( 'Left', 'elementor' ),
						'icon' => 'eicon-h-align-left',
					],
					'yes' => [
						'title' => esc_html__( 'Right', 'elementor' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
			]
		);		

		$this->end_controls_section();

		//title typography
		$this->start_controls_section(
			'title-typography',
			[
				'label' => esc_html__( 'Title Typography', 'control-elementor' ),
			]
		);
		$this->register_title_content_controls( ['prefix' => 'title-'] );

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$gallery = [];
		if(!empty($settings['gallery'])){
			foreach($settings['gallery'] as $key => $value){
				if(!isset($value['id']) || empty($value['id'])) continue;
				$gallery[] = $value['id'];
			}
		}

		$args = array(
			'gallery_display' => $settings['position'],
			'large_img_size' => $settings['large_img_size'],
			'thumb_img_size' => $settings['thumb_img_size'],
			'tag' => $settings['title-size'],
			'title' => $settings['title'], 
			'add_underline' => $settings['title-add_underline'],
			'style' => $settings['title-underline_style'],
			'gallery' => $gallery,
			'footer_text' => $settings['footer_text'],
			'content' => $settings['content'],
		);

		if(locate_template('template-parts/elements/about-company.php')){
			get_template_part('template-parts/elements/about', 'company', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
