<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

/**
 * Elementor heading widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Company_Summary extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'company_summary';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve heading widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Company Summary', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve heading widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-single-post';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'content', 'text' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Company History', 'control-elementor' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Lead text', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => sprintf(__('<b>Founded in 1980,</b> Investment Expert is one of the largest and most experienced global private equity firms. We have established a team investment professionals, focused on buyouts and growth equity investments.', 'control-elementor')),
				'label_block' => true
			]
		);

		$this->add_control(
			'content',
			[
				'label' => esc_html__( 'Description', 'control-elementor' ),
				'type' => Controls_Manager::WYSIWYG,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => sprintf(__('<b>Founded in 1980,</b> Investment Expert is one of the largest and most experienced global private equity firms. We have established a team investment professionals, focused on buyouts and growth equity investments.', 'control-elementor')),
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => esc_html__( 'Lead text alignment', 'control-elementor' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Left', 'control-elementor' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'control-elementor' ),
						'icon' => 'eicon-text-align-center',
					],
					'end' => [
						'title' => esc_html__( 'Right', 'control-elementor' ),
						'icon' => 'eicon-text-align-right',
					]
				],
				'default' => 'left'
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		//figure block tab
		$this->start_controls_section(
			'figure-block',
			[
				'label' => esc_html__( 'Figure Block', 'control-elementor' ),
			]
		);

		$this->add_control(
			'hide_label',
			[
				'label' => esc_html__( 'Hide label area?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'description' => __( 'If yes, label area will be hide in Company summary.', 'control-elementor' ),
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		//label area
		$this->add_control(
			'label_title',
			[
				'label' => esc_html__( 'Label title', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_attr__('Middle-market private equity', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_label' => '',
				), 
			]
		);

		$this->add_control(
			'large_label_text',
			[
				'label' => esc_html__( 'Large Label text', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('EWT', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_label' => '',
				), 
			]
		);

		$this->add_control(
			'label_description',
			[
				'label' => esc_html__( 'Label description', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('{12} Offices in {11} Countries', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_label' => '',
				), 
			]
		);

		//value area
		$this->add_control(
			'hide_value',
			[
				'label' => esc_html__( 'Hide value area?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'description' => __( 'If yes, value area will be hide in Company summary.', 'control-elementor' ),
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		$this->add_control(
			'value_title',
			[
				'label' => esc_html__( 'Value title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter title here', 'control-elementor' ),
				'default' => esc_attr__('Invested', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_value' => '',
				), 
			]
		);

		$this->add_control(
			'large_value_text',
			[
				'label' => esc_html__( 'Value large text', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('$2.4', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_value' => '',
				), 
			]
		);

		$this->add_control(
			'value_unit',
			[
				'label' => esc_html__( 'Value unit', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('B', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_value' => '',
				), 
			]
		);

		//styling
		$this->add_control(
			'gap',
			[
				'label' => esc_html__( 'Horizontal gap', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					''   => __( 'Default', 'control-elementor' ) ,
					'5'  => __( '5px', 'control-elementor' ),
					'10' => __( '10px', 'control-elementor' ),
					'15' => __( '15px', 'control-elementor' ) ,
					'20' => __( '20px', 'control-elementor' ),
					'25' => __( '25px', 'control-elementor' ),
					'30' => __( '30px', 'control-elementor' ),
					'40' => __( '40px', 'control-elementor' ),
					'45' => __( '45px', 'control-elementor' ),
					'50' => __( '50px', 'control-elementor' ),
				],
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$align = '';
		if($settings['align']){
			$align = 'text-' . $settings['align'];
		}

		$args = array(
			'lead_text_align' => $align,
			'lead_text' => $settings['title'],
			'content' => $settings['content'],
			'hide_value' => $settings['hide_value'],
			'hide_label' => $settings['hide_label'],
			'label_title' => $settings['label_title'],
			'label_large' => $settings['large_label_text'],
			'label_desc' => $settings['label_description'],
			'value_title' => $settings['value_title'],
			'value_large' => $settings['large_value_text'],
			'value_unit' => $settings['value_unit'],
			'horizontal_gap' => $settings['gap']
		);

		if(locate_template('template-parts/elements/company-summary.php')){
			get_template_part('template-parts/elements/company', 'summary', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
