<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Controls_Manager\REPEATER;

/**
 * Elementor Contact_Info widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Contact_Info extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'contact_info';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Contact Info', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-mail';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'content', 'image' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Contact Info', 'control-elementor' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_html__( 'Get in Touch', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__( 'Choose Image', 'control-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'description' => esc_attr__('Optional', 'control-elementor'),
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_control(
			'address',
			[
				'label' => esc_html__( 'Address', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your address here', 'control-elementor' ),
				'default' => esc_html__( '70 W. Madison Street, Ste. 1400 Chicago, IL 60602', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'email',
			[
				'label' => esc_html__( 'Email', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Multiple values are comma(,) separated', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your email', 'control-elementor' ),
				'default' => esc_html__( 'info@themeperch.net', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'phone',
			[
				'label' => esc_html__( 'Phone', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter your phone number', 'control-elementor' ),
				'default' => esc_html__( '(214) 550-0405', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'fax',
			[
				'label' => esc_html__( 'Fax', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Multiple values are comma(,) separated', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your fax id', 'control-elementor' ),
				'default' => esc_html__( '(290) 580-9595', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'contact-list-items',
			[
				'label' => esc_html__( 'List Info', 'control-elementor' ),
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'list_title',
			[
				'label' => esc_html__( 'List title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'list_info',
			[
				'label' => esc_html__( 'List info', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter content here', 'control-elementor' ),
				'label_block' => true
			]
		);
		$this->add_control(
			'contact_list_info',
			[
				'label' => esc_attr__('List info', 'control-elementor'),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => []  
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$args = array(
			'image' => $settings['image']['id'],
			'title' => $settings['title'],
			'address' => $settings['address'],
			'phone' => $settings['phone'],
			'fax' => $settings['fax'],
			'email' => $settings['email'],
			'list_info' => $settings['contact_list_info'],
		);

		if(locate_template('template-parts/elements/contact-info.php')){
			get_template_part('template-parts/elements/contact', 'info', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
