<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Control_Media;
use Elementor\Group_Control_Css_Filter;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Controls_Manager\REPEATER;
use controlElementor\Widgets\Traits\Button_Trait;

/**
 * Elementor image box widget.
 *
 * Elementor widget that displays an image, a headline and a text.
 *
 * @since 1.0.0
 */
class Events extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve image box widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'events';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve image box widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Events', 'elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image box widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-calendar';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'image', 'date', 'time', 'booking' ];
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the button widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}


	/**
	 * Register image box widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_image',
			[
				'label' => esc_html__( 'Events', 'elementor' ),
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'event_day',
			[
				'label' => esc_html__( 'Event day', 'elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter day', 'elementor' ),
				'separator' => 'before',
			]
		);
		$repeater->add_control(
			'event_title',
			[
				'label' => esc_html__( 'Event name', 'elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter event name', 'elementor' ),
				'separator' => 'before',
			]
		);
		$repeater->add_control(
			'event_desc',
			[
				'label' => esc_html__( 'Event short description', 'elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter event short description', 'elementor' ),
				'separator' => 'before',
			]
		);
		$repeater->add_control(
			'name',
			[
				'label' => esc_html__( 'Guest\'s Name', 'elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'default' => esc_html__( 'John Doe', 'elementor' ),
				'placeholder' => esc_html__( 'Enter guest\'s name', 'elementor' ),
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'image',
			[
				'label' => esc_html__( 'Choose Guest\'s Image', 'elementor' ),
				'type' => Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);
		$repeater->add_control(
			'event_date',
			[
				'label' => esc_html__( 'Event Date', 'elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter event date', 'elementor' ),
				'separator' => 'none',
				'rows' => 10,
			]
		);
		$repeater->add_control(
			'event_time',
			[
				'label' => esc_html__( 'Event time', 'elementor' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'Enter event time', 'elementor' ),
				'separator' => 'before',
			]
		);
		$repeater->add_control(
			'purchase_link',
			[
				'label' => esc_html__( 'Purchase Link', 'elementor' ),
				'type' => Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'https://your-link.com', 'elementor' ),
				'separator' => 'before',
			]
		);
		$repeater->add_control(
			'purchase_link_text',
			[
				'label' => esc_html__( 'Purchase Link Text', 'elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => 'Buy Tickets',
			]
		);

		$this->add_control(
			'events',
			[
				'label' => 'Events',
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => []  
			]
		);

		$this->add_control(
			'css_class',
			[
				'label' => esc_html__( 'CSS class', 'elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => ''
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render image box widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$name_html = '';
		$designation_html = '';
		$image_html = '';
		$social_links_html = '';

		$has_content = ! Utils::is_empty( $settings['name'] ) || ! Utils::is_empty( $settings['designation'] );

		$this->add_render_attribute( 'team', 'class', 'elementor-team-wrapper card card-team' );
		if ( ! empty( $settings['css_class'] ) ) {
			$this->add_render_attribute( 'team', 'class', $settings['css_class'] );
		}

		$html_before = '<div '.$this->get_render_attribute_string( 'team' ).'>';
		$html_after = '</div>';

		if ( ! empty( $settings['image']['url'] ) ) {
			$this->add_render_attribute( 'image', 'src', $settings['image']['url'] );
			$this->add_render_attribute( 'image', 'alt', Control_Media::get_image_alt( $settings['image'] ) );
			$this->add_render_attribute( 'image', 'title', Control_Media::get_image_title( $settings['image'] ) );

			$image_html = wp_kses_post( Group_Control_Image_Size::get_attachment_image_html( $settings, 'thumbnail', 'image' ) );

		}

		if ( $has_content ) {

			if ( ! Utils::is_empty( $settings['name'] ) ) {
				$this->add_render_attribute( 'name', 'class', 'elementor-team-name' );

				$this->add_inline_editing_attributes( 'name', 'none' );

				$title_html = $settings['name'];

				$name_html = sprintf( '<h5 %1$s>%2$s</h5>', $this->get_render_attribute_string( 'name' ), $title_html );
			}

			if ( ! Utils::is_empty( $settings['designation'] ) ) {
				$this->add_render_attribute( 'designation', 'class', 'elementor-team-designation' );

				$this->add_inline_editing_attributes( 'designation' );

				$designation_html = sprintf( '<p %1$s>%2$s</p>', $this->get_render_attribute_string( 'designation' ), $settings['designation'] );
			}

		}

		if( $settings['link_facebook']['url'] || $settings['link_linkedin']['url'] || $settings['link_twitter']['url'] ){
			$social_links_html = '<div class="social-btn d-flex gap-2 w-100 justify-content-center">';

			if($settings['link_facebook']['url']){
				$this->add_link_attributes( 'link_facebook', $settings['link_facebook'] );
				$social_links_html .= '<a ' . $this->get_render_attribute_string( 'link_facebook' ) . '><i class="bi bi-facebook"></i></a>';
			}

			if($settings['link_linkedin']['url']){
				$this->add_link_attributes( 'link_linkedin', $settings['link_linkedin'] );
				$social_links_html .= '<a ' . $this->get_render_attribute_string( 'link_linkedin' ) . '><i class="bi bi-linkedin"></i></a>';
			}

			if($settings['link_twitter']['url']){
				$this->add_link_attributes( 'link_twitter', $settings['link_twitter'] );
				$social_links_html .= '<a ' . $this->get_render_attribute_string( 'link_twitter' ) . '><i class="bi bi-twitter"></i></a>';
			}

			$social_links_html .= '</div>';
		}

		//render html
		$team_html = $html_before .'<div class="card-img rounded-4 position-relative">
			<div class="overlay overlay-dark"></div>' . $image_html .
			$social_links_html .
		'</div>' .
		'<div class="card-body">' .
		$name_html . $designation_html .
		'</div>' .
		$html_after;

		Utils::print_unescaped_internal_string( $team_html );
	}

}
