<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use controlElementor\Widgets\Traits\Text_Trait;

/**
 * Elementor feature widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Featured_Box extends Widget_Base {

	use Text_Trait;

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'featured_box';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Featured Box', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-kit-parts';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'content', 'image' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Featured Box', 'control-elementor' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Heading title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_html__( 'Increase Profitability', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'content',
			[
				'label' => esc_html__( 'Description', 'control-elementor' ),
				'type' => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_html__( 'Add Your Heading Text Here', 'control-elementor' ),
			]
		);

		$this->add_control(
			'background_color',
			[
				'label' => esc_html__( 'Background type', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'bg_gray' => esc_attr__('Default', 'control-elementor'),
					'primary-bg' => esc_attr__('Primary color', 'control-elementor'),
					'secondary-bg-dark' => esc_attr__('Secondary color', 'control-elementor'),
				],
				'default' => 'bg_gray',
			]
		);

		$this->add_control(
			'use_image',
			[
				'label' => esc_html__( 'Use custom featured image?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__( 'Choose Image', 'elementor' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => array(
					'use_image' => 'yes',
				),
			]
		);

		$this->add_control(
			'position',
			[
				'label' => esc_html__( 'Image Position', 'elementor' ),
				'type' => Controls_Manager::CHOOSE,
				'default' => '',
				'options' => [
					'order-sm-0' => [
						'title' => esc_html__( 'Left', 'elementor' ),
						'icon' => 'eicon-h-align-left',
					],
					'order-sm-2' => [
						'title' => esc_html__( 'Right', 'elementor' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'condition' => array(
					'use_image' => 'yes',
				),
				'toggle' => false,
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		//title typography
		$this->start_controls_section(
			'title-typography',
			[
				'label' => esc_html__( 'Title Typography', 'control-elementor' ),
			]
		);
		$this->register_title_content_controls( ['prefix' => 'title-'] );

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$args = array(
			'bg_type' => $settings['background_color'],
			'add_custom_image' => $settings['use_image'],
			'image_right' => $settings['position'],
			'tag' => $settings['title-size'],
			'title' => $settings['title'], 
			'add_underline' => $settings['title-add_underline'],
			'style' => $settings['title-underline_style'],
			'content' => $settings['content']
		);

		if(isset($settings['image']['url'])){
			$settings['image'] = $settings['image']['url'];
		}

		if(locate_template('template-parts/elements/featured-box.php')){
			get_template_part('template-parts/elements/featured-box', '', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
