<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

/**
 * Elementor Figures_Block widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Figures_Block extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'figures_block';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve heading widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Figures Block', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve heading widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-parallax';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'content', 'text' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Figures Block', 'control-elementor' ),
			]
		);

		$this->add_control(
			'type',
			[
				'label' => esc_html__( 'Display in a row?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'description' => __( 'Display in a row or up-down style', 'control-elementor' ),
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		$this->add_control(
			'hide_label',
			[
				'label' => esc_html__( 'Hide label area?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'description' => __( 'If yes, label area will be hide in Company summary.', 'control-elementor' ),
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		//label area
		$this->add_control(
			'label_title',
			[
				'label' => esc_html__( 'Label title', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_attr__('Middle-market private equity', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_label' => '',
				), 
			]
		);

		$this->add_control(
			'large_label_text',
			[
				'label' => esc_html__( 'Large Label text', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('EWT', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_label' => '',
				), 
			]
		);

		$this->add_control(
			'label_description',
			[
				'label' => esc_html__( 'Label description', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('{12} Offices in {11} Countries', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_label' => '',
				), 
			]
		);

		//value area
		$this->add_control(
			'hide_value',
			[
				'label' => esc_html__( 'Hide value area?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'description' => __( 'If yes, value area will be hide in Company summary.', 'control-elementor' ),
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		$this->add_control(
			'value_title',
			[
				'label' => esc_html__( 'Value title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter title here', 'control-elementor' ),
				'default' => esc_attr__('Invested', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_value' => '',
				), 
			]
		);

		$this->add_control(
			'large_value_text',
			[
				'label' => esc_html__( 'Value large text', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('$2.4', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_value' => '',
				), 
			]
		);

		$this->add_control(
			'value_unit',
			[
				'label' => esc_html__( 'Value unit', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'default' => esc_attr__('B', 'control-elementor'),
				'label_block' => true,
				'condition' => array(
					'hide_value' => '',
				), 
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$args = array(
			'type' => $settings['type'],
			'label_title' => $settings['label_title'],
			'label_large' => $settings['large_label_text'],
			'label_desc' => $settings['label_description'],
			'value_title' => $settings['value_title'],
			'value_large' => $settings['large_value_text'],
			'value_unit' => $settings['value_unit'],
			'hide_value' => $settings['hide_value'],
			'hide_label' => $settings['hide_label'],
		);

		if(locate_template('template-parts/elements/figures-block.php')){
			get_template_part('template-parts/elements/figures', 'block', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
