<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use controlElementor\Widgets\Traits\Text_Trait;

/**
 * Elementor Header_Content widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Header_Content extends Widget_Base {

	use Text_Trait;

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'header_content';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve heading widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Header Content', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve heading widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-header';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'content', 'text' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Header_Content', 'control-elementor' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Heading title', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_html__( 'Investment Expert is one of the largest and most experienced global private equity investors', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'subtitle',
			[
				'label' => esc_html__( 'Heading subtitle', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_html__( 'Middle-market private equity', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'content',
			[
				'label' => esc_html__( 'Description', 'control-elementor' ),
				'type' => Controls_Manager::WYSIWYG,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
			]
		);

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Button Link', 'control-elementor' ),
				'type' => Controls_Manager::URL,
				'placeholder' => esc_html__( 'https://your-link.com', 'control-elementor' ),
				'default' => [
					'url' => '#'
				],
				'separator' => 'before',
			]
		);

		$this->add_control(
			'link_text',
			[
				'label' => esc_html__( 'Button Text', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Find out more', 'control-elementor' ),
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style-settings',
			[
				'label' => esc_html__( 'Style', 'control-elementor' ),
			]
		);

		$this->add_control(
			'style',
			[
				'label' => esc_html__( 'Style', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'style1' => esc_attr__('Style 1', 'control-elementor'),
					'style2' => esc_attr__('Style 2', 'control-elementor'),
					'style3' => esc_attr__('Style 3', 'control-elementor'),
				],
				'default' => 'style1',
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => esc_html__( 'Alignment', 'control-elementor' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Left', 'control-elementor' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'control-elementor' ),
						'icon' => 'eicon-text-align-center',
					],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title-typography',
			[
				'label' => esc_html__( 'Title Typography', 'control-elementor' ),
			]
		);
		$this->register_title_content_controls( ['prefix' => 'title-'] );

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$align = '';
		if(!empty($settings['align'])){
			$align = 'text-'.$settings['align'];
		}

		$args = array(
			'style' => $settings['style'],
			'align' => $align,
			'title_tag' => $settings['title-size'],
			'title_font_size' => $settings['title-size_class'],
			'title' => $settings['title'], 
			'subtitle' => $settings['subtitle'],
			'desc' => $settings['content'],
			'button_text' => $settings['link_text'],
			'button_url' => $settings['link']['url'],
		);

		if(locate_template('template-parts/elements/header-content.php')){
			get_template_part('template-parts/elements/header', 'content', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
