<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Controls_Manager\REPEATER;

/**
 * Elementor Header_Info widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Header_Info extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'header_info';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Onepage Header Info', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-logo';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'image', 'icon' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Onepage Header Info', 'control-elementor' ),
			]
		);

		$this->add_control(
			'logo',
			[
				'label' => esc_html__( 'Navbar logo', 'control-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'icon',
			[
				'label' => esc_html__( 'Icon', 'control-elementor' ),
				'type' => Controls_Manager::ICONS,			
			]
		);
		$repeater->add_control(
			'link',
			[
				'label' => esc_html__( 'Link', 'control-elementor' ),
				'type' => Controls_Manager::URL,
				'placeholder' => esc_html__( 'https://your-link.com', 'control-elementor' ),
				'default' => [
					'url' => '#'
				],
				'label_block' => true
			]
		);
		
		$this->add_control(
			'social_links',
			[
				'label' => esc_attr__('Social links', 'control-elementor'),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'title' => esc_attr__('Facebook', 'control-elementor'),
						'icon' => [
							'value' => 'fab fa-facebook',
							'library' => 'fa-brands',
						],
						'link' => '#',
					],
					[
						'title' => esc_attr__('Twitter', 'control-elementor'),
						'icon' => [
							'value' => 'fab fa-twitter',
							'library' => 'fa-brands',
						],
						'link' => '#',
					],
					[
						'title' => esc_attr__('Linkedin', 'control-elementor'),
						'icon' => [
							'value' => 'fab fa-linkedin',
							'library' => 'fa-brands',
						],
						'link' => '#',
					],
					[
						'title' => esc_attr__('RSS', 'control-elementor'),
						'icon' => [
							'value' => 'fas fa-rss',
							'library' => 'fa-solid',
						],
						'link' => '#',
					]
				]  
			]
		);	

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$social_links = $settings['social_links'];
		if(!empty($social_links)){
			foreach($social_links as $key => $link){
				if(!isset($link['icon'])) continue;
				if(!isset($link['icon']['value']) || empty($link['icon']['value'])) continue;

				$social_links[$key]['icon'] = $link['icon']['value'];
				$social_links[$key]['link'] = $link['link']['url'];
			}
		}

		$args = array(
			'nav_logo'  => $settings['logo']['url'],
    		'socaial_links' => $social_links,
		);

		if(locate_template('template-parts/elements/onepage-header.php')){
			get_template_part('template-parts/elements/onepage', 'header', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
