<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

/**
 * Elementor List_Item widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class List_Item extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'list_item';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'List Item', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-radio';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'content', 'icon', 'link' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'List_Item', 'control-elementor' ),
			]
		);

		$this->add_control(
			'add_icon',
			[
				'label' => esc_html__( 'Add icon?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		$this->add_control(
			'icon',
			[
				'label' => esc_html__( 'Icon', 'control-elementor' ),
				'type' => Controls_Manager::ICONS,	
				'default' => [
					'value' => 'perch perch-integrity',
					'library' => 'perchicons',
				],
				'condition' => [
					'add_icon' => 'yes'
				]		
			]
		);

		$this->add_control(
			'icon_bg',
			[
				'label' => esc_html__( 'Icon background?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
				'condition' => [
					'add_icon' => 'yes'
				]
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Use {} to highlight text', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'default' => esc_html__( 'Investment Deal Team', 'control-elementor' ),
				'label_block' => true
			]
		);

		$this->add_control(
			'style',
			[
				'label' => esc_html__( 'Title style', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'normal' => esc_attr__('Normal', 'control-elementor'),
					'h_small' => esc_attr__('Bold', 'control-elementor'),
					'h4' => esc_attr__('H4', 'control-elementor'),
					'h5' => esc_attr__('H5', 'control-elementor'),
					'h6' => esc_attr__('H6', 'control-elementor'),
				],
				'default' => 'h_small'
			]
		);

		$this->add_control(
			'add_link',
			[
				'label' => esc_html__( 'Add link?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		$this->add_control(
			'link',
			[
				'label' => esc_html__( 'Title Link', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => '#',
				'condition' => [
					'add_link' => 'yes'
				]
			]
		);

		$this->add_control(
			'content',
			[
				'label' => esc_html__( 'Description', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'description' => esc_html__( 'Leave blank to avoid this area', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$args = array(
			'style' => $settings['style'],
			'title' => $settings['title'],
			'add_icon' => $settings['add_icon'],
			'icon_bg' => $settings['icon_bg'],
			'add_link' => $settings['add_link'],
			'title_link' => $settings['link'],
			'desc' => $settings['content'],
		);

		if(isset($settings['icon']['library'])){
			$args['type'] = $settings['icon']['library'];
			$args['icon_'.$settings['icon']['library']] = $settings['icon']['value'];
		}



		if(locate_template('template-parts/elements/list-item.php')){
			get_template_part('template-parts/elements/list', 'item', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
