<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Controls_Manager\REPEATER;

/**
 * Elementor Nav_Menu widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Nav_Menu extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'nav_menu';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Onepage Nav Menu', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-post-slider';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'image' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Onepage Nav Menu', 'control-elementor' ),
			]
		);

		$this->add_control(
			'navbar_visible',
			[
				'label' => esc_html__( 'Navbar visible?', 'control-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
			]
		);

		$this->add_control(
			'nav_logo',
			[
				'label' => esc_html__( 'Logo dark', 'control-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'description' => esc_html__('Display only on sticky navbar', 'control-elementor'),
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_control(
			'header_logo',
			[
				'label' => esc_html__( 'Logo light', 'control-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'description' => esc_html__('Display only on header', 'control-elementor'),
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'navbar_visible' => 'yes'
				]
			]
		);

		$this->add_control(
			'nav_menu',
			[
				'label' => esc_html__( 'Select nav menu', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'description' => esc_html__('You can create menu Appearance > Menu', 'investment'),
				'options' => control_elementor_get_terms('nav_menu', 'slug'),
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'nav_button_settings',
			[
				'label' => esc_html__( 'Nav Buttons', 'control-elementor' ),
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'button_text',
			[
				'label' => esc_html__( 'Button text', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter text here', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'button_url',
			[
				'label' => esc_html__( 'Button Link', 'control-elementor' ),
				'type' => Controls_Manager::URL,
				'placeholder' => esc_html__( 'https://your-link.com', 'control-elementor' ),
				'default' => [
					'url' => '#'
				],
			]
		);
		$this->add_control(
			'buttons',
			[
				'label' => esc_attr__('Nav buttons', 'control-elementor'),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'title_field' => '{{{ button_text }}}',
				'default' => [
					[
						'button_text' => 'Get started',
						'button_url' => '#pricing',
					]
				]  
			]
		);	

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$buttons = $settings['buttons'];
		if(!empty($buttons)){
			foreach($buttons as $key => $link){
				$buttons[$key]['button_url'] = $link['button_url']['url'];
			}
		}

		$args = array(
			'navbar_visible' => $settings['navbar_visible'],
			'nav_logo'  => $settings['nav_logo']['url'],
			'nav_menu' => $settings['nav_menu'],
			'buttons' => $buttons
		);

		if(isset($settings['header_logo'])){
			$args['header_logo'] =  $settings['header_logo']['url'];
		}

		if(locate_template('template-parts/elements/onepage-nav.php')){
			get_template_part('template-parts/elements/onepage', 'nav', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
