<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Controls_Manager\REPEATER;

/**
 * Elementor OpenStreet_Map widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class OpenStreet_Map extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'openstreet_map';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'OpenStreet Map', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-map-pin';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'image', 'map' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'OpenStreet Map', 'control-elementor' ),
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'style',
			[
				'label' => esc_html__( 'Style', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'default' => esc_attr__('Default', 'control-elementor'),
					'embaed' => esc_attr__('Iframe', 'control-elementor'),
				],
				'label_block' => true
			]
		);
		$repeater->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'lat',
			[
				'label' => esc_html__( 'Latitude', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter latitude', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'long',
			[
				'label' => esc_html__( 'Longitude', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter longitude', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'zoom',
			[
				'label' => esc_html__( 'Zoom', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter zoom value', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'marker',
			[
				'label' => esc_html__( 'Marker', 'control-elementor' ),
				'type' => Controls_Manager::MEDIA,
			]
		);
		$this->add_control(
			'maps_info',
			[
				'label' => esc_attr__('Maps info', 'control-elementor'),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'style' => 'default',
						'title' => 'Chicago Office',
						'lat' => 41.506146,
						'long' => -87.6355995,
						'zoom' => 12,						                    
						'marker' => [
							'url' => CONTROL_ELEMENTOR_ASSETS.'/images/map-marker.png'
						]
					],
					[
						'style' => 'default',
						'title' => 'Washington Office',	
						'lat' => 38.9071923,
						'long' => -77.0368707,
						'zoom' => 12,				                    				
						'marker' => [
							'url' => CONTROL_ELEMENTOR_ASSETS.'/images/map-marker.png'
						]            
					],
					[
						'style' => 'default',
						'title' => 'New York Office',
						'lat' => 40.7127837,
						'long' => -74.0059413,
						'zoom' => 12,
						'marker' => [
							'url' => CONTROL_ELEMENTOR_ASSETS.'/images/map-marker.png'
						]	                    
					],
					[
						'style' => 'default',
						'title' => 'London Office',
						'lat' => 51.5073509,
						'long' => -0.1277583,
						'zoom' => 12,						                    
						'marker' => [
							'url' => CONTROL_ELEMENTOR_ASSETS.'/images/map-marker.png'
						]						                    
					]
				]  
			]
		);	

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$maps_info = $settings['maps_info'];
		if(!empty($maps_info)){
			foreach($maps_info as $key => $info){
				if(isset($info['marker']['url'])){
					$maps_info[$key]['marker'] = $info['marker']['url'];
				}
			}
		}

		$args = array(
			'mapsinfo' => $maps_info
		);

		if(locate_template('template-parts/elements/openstreet-map.php')){
			get_template_part('template-parts/elements/openstreet', 'map', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
