<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use controlElementor\Widgets\Traits\WP_Query_Trait;

/**
 * Elementor Post_Template widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Post_Template extends Widget_Base {

	use WP_Query_Trait;

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'post_template';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Post Template', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-post-slider';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {

		$category = control_elementor_get_terms('category', 'slug');
		$category = (!$category)? array() : $category;

		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Posts Query', 'control-elementor' ),
			]
		);

		$this->register_query_arguments_controls( 
			[
				'prefix' => 'post_',
				'post_type' => 'post',
				'taxonomy' => 'category',
			] 
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'portfolio_settings_title',
			[
				'label' => esc_html__( 'General Settings', 'control-elementor' ),
			]
		);

		$this->add_control(
			'title_display',
			[
				'label' => esc_html__( 'Display Title', 'control-elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Our Sector Knowledge', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'label_block' => true,
				'condition' => [
					'title_display' => 'yes',
				],
			]
		);

		$this->add_control(
			'desc',
			[
				'label' => esc_html__( 'Description', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'default' => esc_html__( 'Advent has one of the world’s largest and most experienced private equity teams, with more than 170 investment professionals across four continents.', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your description', 'control-elementor' ),
				'label_block' => true,
				'condition' => [
					'title_display' => 'yes',
				],
			]
		);

		$this->add_control(
			'button_link_text',
			[
				'label' => esc_html__( 'Button link text', 'control-elementor' ),
				'description' => esc_html__( 'Leave blank to avoid button', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'default' => esc_html__( 'View all news', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter button link text', 'control-elementor' ),
				'label_block' => true,
				'condition' => [
					'title_display' => 'yes',
				],
			]
		);

		$this->add_control(
			'image_link_type',
			[
				'label' => esc_html__( 'Image link type', 'control-elementor' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'popup' => esc_html__( 'Popup', 'control-elementor' ),
					'link' => esc_html__( 'Post link', 'control-elementor' ),
				]
			]
		);

		$this->add_control(
			'template',
			[
				'label' => esc_html__( 'Display type', 'control-elementor' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'templates/default-loop-1.php' => esc_html__( 'Default', 'control-elementor' ),
					//'templates/isotope-loop-1.php' => esc_html__( 'Isotope', 'control-elementor' ),
				]
			]
		);

		$this->add_control(
			'column',
			[
				'label' => esc_html__( 'Column', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 4,
				'step' => 1,
				'default' => 3,
			]
		);

		$this->add_control(
			'active',
			[
				'label' => esc_html__( 'Active Category', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'label_block' => true,
				'description' => 'Select categories to set as active tab',
				'options' => array_merge(array('' => 'All'), $category),
				'condition' => [
					'template' => 'templates/isotope-loop-1.php',
				],
			]
		);

		$this->add_control(
			'read_more',
			[
				'label' => esc_html__( 'Category button title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'know more', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter button title', 'control-elementor' ),
				'label_block' => true,
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$query_args = [
			'post_type' => 'post',
			'posts_per_page' => $settings['post_posts_per_page'],
			'id' => $settings['post_id'],
			'tax_term' => $settings['post_tax_term'],
			'tax_operator' => $settings['post_operator'],
			'order' => $settings['post_order'],
			'orderby' => $settings['post_orderby'],
		];

		$args = [
			'query_args' => $query_args,
			'info' => $settings
		];

		$template = locate_template( $settings['template'] );

		// Only allow templates that are in the active theme directory, parent theme
		// directory, or the /wp-includes/theme-compat/ directory (prevent directory 
		// traversal attacks).
		$template_in_theme_or_parent_theme_or_compat = (
			// Template is in current theme folder.
			0 === strpos( realpath( $template ), realpath( STYLESHEETPATH ) ) ||
			// Template is in current or parent theme folder.
			0 === strpos( realpath( $template ), realpath( TEMPLATEPATH ) ) ||
			// Template is in theme-compat folder.
			0 === strpos( realpath( $template ), realpath( ABSPATH . WPINC . '/theme-compat/' ) )
		);

		if ( $template_in_theme_or_parent_theme_or_compat ) {
			require_once( $template );
		}
		
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
