<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Controls_Manager\REPEATER;

/**
 * Elementor Skillbar widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Skillbar extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'skillbar';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Skillbar', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-skill-bar';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'color' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Skillbar', 'control-elementor' ),
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'label',
			[
				'label' => esc_html__( 'Label', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter label', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'value',
			[
				'label' => esc_html__( 'Value', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'description' => esc_html__( 'Enter value of bar.', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter label', 'control-elementor' ),
				'label_block' => true
			]
		);
		$repeater->add_control(
			'customcolor',
			[
				'label' => esc_html__( 'Text Color', 'textdomain' ),
				'type' => Controls_Manager::COLOR,
				'description' => esc_html__( 'Select custom single bar background color.', 'control-elementor' ),
			]
		);
		
		$this->add_control(
			'skills',
			[
				'label' => esc_attr__('Skills', 'control-elementor'),
				'type' => Controls_Manager::REPEATER,
				'description' => esc_attr__( 'Enter values for graph - value, title and color.', 'control-elementor' ),
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'label' => __( 'Development', 'control-elementor' ),
						'value' => '90',
						'customcolor' => '#ffc000'
					],
					[
						'label' => __( 'Design', 'control-elementor' ),
						'value' => '80',
						'customcolor' => '#fc7f0c'
					],
					[
						'label' => __( 'Photography', 'control-elementor' ),
						'value' => '85',
						'customcolor' => ot_get_option( 'primary_color', '#f3525a' )
					],
					[
						'label' => __( 'Marketing', 'control-elementor' ),
						'value' => '70',
						'customcolor' => ot_get_option( 'secondary_color', '#20304f' )
					]
				]  
			]
		);	

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$args = array(
			'values' => $settings['skills'],
		);

		if(locate_template('template-parts/elements/skillbar.php')){
			get_template_part('template-parts/elements/skillbar', '', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
