<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use controlElementor\Widgets\Traits\WP_Query_Trait;

/**
 * Elementor Team_Template widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Team_Template extends Widget_Base {

	use WP_Query_Trait;

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'team_template';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Team Template', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-post-slider';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Posts Query', 'control-elementor' ),
			]
		);

		$this->register_query_arguments_controls(
			[
				'prefix' => 'team_',
				'post_type' => 'team',
				'taxonomy' => '',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'team_settings_title',
			[
				'label' => esc_html__( 'General Settings', 'control-elementor' ),
			]
		);

		$this->add_control(
			'title_display',
			[
				'label' => esc_html__( 'Display Title', 'control-elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( '{Experienced} Team', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your title', 'control-elementor' ),
				'label_block' => true,
				'condition' => [
					'title_display' => 'yes',
				],
			]
		);

		$this->add_control(
			'desc',
			[
				'label' => esc_html__( 'Description', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'default' => esc_html__( 'Advent has one of the world’s largest and most experienced private equity teams, with more than 170 investment professionals across four continents.', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter your description', 'control-elementor' ),
				'label_block' => true,
				'condition' => [
					'title_display' => 'yes',
				],
			]
		);

		$this->add_control(
			'button_link_text',
			[
				'label' => esc_html__( 'Button link text', 'control-elementor' ),
				'description' => esc_html__( 'Leave blank to avoid button', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'default' => esc_html__( 'Meet our team', 'control-elementor' ),
				'placeholder' => esc_html__( 'Enter button link text', 'control-elementor' ),
				'label_block' => true,
				'condition' => [
					'title_display' => 'yes',
				],
			]
		);

		$this->add_control(
			'column',
			[
				'label' => esc_html__( 'Column', 'textdomain' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 4,
				'step' => 1,
				'default' => 3,
			]
		);

		$this->add_control(
			'disable_more_button',
			[
				'label' => esc_html__( 'Disable more link button', 'control-elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'disable_more_link',
			[
				'label' => esc_html__( 'Disable more link', 'control-elementor' ),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'control-elementor' ),
				'label_off' => esc_html__( 'No', 'control-elementor' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$settings['posts_per_page'] = $settings['team_posts_per_page'];

		if(locate_template('template-parts/elements/teams.php')){
			get_template_part('template-parts/elements/teams', '', $settings);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
