<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

/**
 * Query Template widget.
 *
 * Elementor widget that displays stats and numbers in an escalating manner.
 *
 * @since 1.0.0
 */
class Template extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve Query Template widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'template';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Query Template widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Template', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Query Template widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'dashicons dashicons-wordpress-alt';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'template', 'query', 'post', 'wp_query' ];
	}

	/**
	 * Register Query Template widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'post_type_template',
			[
				'label' => esc_html__( 'Template', 'control-elementor' ),
			]
		);

		$post_types = [
			'post' => esc_html__( 'Post', 'control-elementor' ),
		];
		$post_types = apply_filters('control_elementor/template/post_types', $post_types);

		$this->add_control(
			'post_type',
			[
				'label' => esc_html__( 'Post type', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,				
				'options' => $post_types,
				'label_block' => true,
				'default' => 'post'
			]
		);

		foreach ($post_types as $post_type => $value) {
			$templates = [
				'default.php' => 'Default'
			];
			$templates = apply_filters('control_elementor/'.$post_type.'/templates', $templates);
			$this->add_control(
				$post_type.'_template',
				[
					'label' => sprintf(esc_html__( 'Choose Template for %s', 'control-elementor' ), $value),
					'type' => Controls_Manager::SELECT,
					'label_block' => true,
					'condition' => [
						'post_type' => $post_type,
					],
					'options' => $templates,
				]
			);
		}

		

		
		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'parameters',
			[
				'label' => esc_html__( 'Query Parameters', 'control-elementor' ),
			]
		);
		$this->add_control(
			'posts_per_page',
			[
				'label' => esc_html__( 'Number of post to show per page', 'control-elementor' ),
				'type' => Controls_Manager::NUMBER,				
				'default' => get_option('posts_per_page')
			]
		);
		$this->end_controls_section();

		

		
	}

	

	/**
	 * Render Query Template widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		if( empty($settings['post_type']) )  return;

		$template = $settings['post_type'].'_template';
		if(empty($settings[$template])) return;

		$query_args = [
			'post_type' => $settings['post_type'],
			'posts_per_page' => !empty($settings['posts_per_page'])? $settings['posts_per_page'] : 10
		];

		control_elementor_locate_template($settings[$template], true, true, $query_args);	
		
	}

	
}
