<?php
namespace controlElementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Controls_Manager\REPEATER;

/**
 * Elementor Testimonials widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class Testimonials extends Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve heading widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'testimonials';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve Feature widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Testimonials', 'control-elementor' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve Feature widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-testimonial';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the heading widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 2.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'control-elementor' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'title', 'image' ];
	}

	/**
	 * Register heading widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_title',
			[
				'label' => esc_html__( 'Testimonials', 'control-elementor' ),
			]
		);

		$this->add_control(
			'style',
			[
				'label' => esc_html__( 'Style', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'style1',
				'options' => [
					'style1' => esc_attr__('Style1(Circle image on left side)', 'control-elementor'),
					'style2' => esc_attr__('Style2(Image placed in bootom of testimonial)', 'control-elementor'),
				],
				'label_block' => true
			]
		);

		$this->add_control(
			'bg_class',
			[
				'label' => esc_html__( 'Testimonial Background', 'control-elementor' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'tra-bg',
				'options' => [
					'tra-bg' => esc_attr__('Default Background color', 'control-elementor'),
					'primary-bg' => esc_attr__('Primary Background color', 'control-elementor'),
					'secondary-bg-dark' => esc_attr__('Secondary Background color', 'control-elementor'),
					'secondary-light-bg' => esc_attr__('Secondary Light Background color', 'control-elementor'),
					'secondary-dark-bg' => esc_attr__('Secondary Dark Background color', 'control-elementor'),
					'bg_gray' => esc_attr__('Gray Background color', 'control-elementor')
				],
				'label_block' => true
			]
		);

		$repeater = new \Elementor\Repeater();
		$repeater->add_control(
			'image',
			[
				'label' => esc_html__( 'Testimonial image', 'control-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'label_block' => true
			]
		);

		$repeater->add_control(
			'title',
			[
				'label' => esc_html__( 'Title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'label_block' => true
			]
		);

		$repeater->add_control(
			'subtitle',
			[
				'label' => esc_html__( 'Sub-title', 'control-elementor' ),
				'type' => Controls_Manager::TEXT,
				'label_block' => true
			]
		);

		$repeater->add_control(
			'signature',
			[
				'label' => esc_html__( 'Signature', 'control-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'desc',
			[
				'label' => esc_html__( 'Testimonial description', 'control-elementor' ),
				'type' => Controls_Manager::TEXTAREA,
				'label_block' => true
			]
		);
		
		$this->add_control(
			'testimonials',
			[
				'label' => esc_attr__('Testimonials', 'control-elementor'),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'image' => [ 'url' => INVESTMENT_ASSETS.'/images/photo-man.png' ],
						'title' => 'Alexa Von Tobel',				                    
						'subtitle' => 'CEO Finance Consult Manager',                   
						'desc' => '"Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad',	
						'signature' => [ 'url' => INVESTMENT_ASSETS.'/images/painting-1.png' ],	
					],
				]  
			]
		);	

		$this->add_control(
			'view',
			[
				'label' => esc_html__( 'View', 'control-elementor' ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render heading widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		
		$args = [
			'style' => $settings['style'],
			'bg_class' => $settings['bg_class'],
			'testimonials' => $settings['testimonials']
		];

		if(locate_template('template-parts/elements/testimonials.php')){
			get_template_part('template-parts/elements/testimonials', '', $args);
		}
	}

	/**
	 * Render heading widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 2.9.0
	 * @access protected
	 */
	protected function content_template() {}
}
