<?php
/**
 * Retrieve the name of the highest priority template file that exists.
 *
 * Searches in the STYLESHEETPATH before TEMPLATEPATH and CONTROL_ELEMENTOR_TEMPLATEPATH
 * so that themes which inherit from a parent theme can just overload one file.
 *
 * @since 2.7.0
 * @since 5.5.0 The `$args` parameter was added.
 *
 * @param string|array $template_names Template file(s) to search for, in order.
 * @param bool         $load           If true the template file will be loaded if it is found.
 * @param bool         $require_once   Whether to require_once or require. Has no effect if `$load` is false.
 *                                     Default true.
 * @param array        $args           Optional. Additional arguments passed to the template.
 *                                     Default empty array.
 * @return string The template filename if one is located.
 */
function control_elementor_locate_template( $template_names, $load = false, $require_once = true, $args = array() ) {
	$located = '';
    $templates_dir = apply_filters( 'control_elementor/template_directory', '/' );
	foreach ( (array) $template_names as $template_name ) {
		if ( ! $template_name ) {
			continue;
		}
		if ( file_exists( STYLESHEETPATH . $templates_dir . $template_name ) ) {
			$located = STYLESHEETPATH . $templates_dir . $template_name;
			break;
		} elseif ( file_exists( TEMPLATEPATH . $templates_dir . $template_name ) ) {
			$located = TEMPLATEPATH . $templates_dir . $template_name;
			break;
		} elseif ( file_exists( CONTROL_ELEMENTOR_TEMPLATEPATH . $template_name ) ) {
			$located = CONTROL_ELEMENTOR_TEMPLATEPATH . $template_name;
			break;
		}
	}

	if ( $load && '' !== $located ) {
		load_template( $located, $require_once, $args );
	}

	return $located;
}

/**
 * Loads a template part into a template.
 *
 * Provides a simple mechanism for child themes to overload reusable sections of code
 * in the theme.
 *
 * Includes the named template part for a theme or if a name is specified then a
 * specialised part will be included. If the theme contains no {slug}.php file
 * then no template will be included.
 *
 * The template is included using require, not require_once, so you may include the
 * same template part multiple times.
 *
 * For the $name parameter, if the file is called "{slug}-special.php" then specify
 * "special".
 *
 * @since 3.0.0
 * @since 5.5.0 A return value was added.
 * @since 5.5.0 The `$args` parameter was added.
 *
 * @param string $slug The slug name for the generic template.
 * @param string $name The name of the specialised template.
 * @param array  $args Optional. Additional arguments passed to the template.
 *                     Default empty array.
 * @return void|false Void on success, false if the template does not exist.
 */
function control_elementor_template( $slug, $name = null, $args = array() ) {
	/**
	 * Fires before the specified template part file is loaded.
	 *
	 * The dynamic portion of the hook name, `$slug`, refers to the slug name
	 * for the generic template part.
	 *
	 * @param string      $slug The slug name for the generic template.
	 * @param string|null $name The name of the specialized template.
	 * @param array       $args Additional arguments passed to the template.
	 */
	do_action( "get_template_part_{$slug}", $slug, $name, $args );

	$templates = array();
	$name      = (string) $name;
	if ( '' !== $name ) {
		$templates[] = "{$slug}-{$name}.php";
	}

	$templates[] = "{$slug}.php";

	/**
	 * Fires before an attempt is made to locate and load a template part.
     *
	 * @param string   $slug      The slug name for the generic template.
	 * @param string   $name      The name of the specialized template.
	 * @param string[] $templates Array of template files to search for, in order.
	 * @param array    $args      Additional arguments passed to the template.
	 */
	do_action( 'get_template_part', $slug, $name, $templates, $args );

	if ( ! control_elementor_locate_template( $templates, true, false, $args ) ) {
		return false;
	}
}

if( !function_exists('control_elementor_get_terms') ){
	function control_elementor_get_terms( $tax = 'category', $key = 'id' ) {
		$terms = array();
	
		if(!taxonomy_exists($tax)) return false;
	
		if ( $key === 'id' ) foreach ( (array) get_terms( $tax, array( 'hide_empty' => false ) ) as $term ) $terms[$term->term_id] = $term->name;
		  elseif ( $key === 'slug' ) foreach ( (array) get_terms( $tax, array( 'hide_empty' => false ) ) as $term ) $terms[$term->slug] = $term->name;
			return $terms;
	}
}

/**
 * Get size information for all currently-registered image sizes.
 *
 * @global $_wp_additional_image_sizes
 * @uses   get_intermediate_image_sizes()
 * @return array $sizes Data for all currently-registered image sizes.
 */
function control_elementor_get_image_sizes() {
	global $_wp_additional_image_sizes;
	
	$sizes = array();
	
	foreach ( get_intermediate_image_sizes() as $_size ) {
		if ( in_array( $_size, array(
			 'thumbnail',
			'medium',
			'medium_large',
			'large' 
		) ) ) {
			$sizes[ $_size ][ 'width' ]  = get_option( "{$_size}_size_w" );
			$sizes[ $_size ][ 'height' ] = get_option( "{$_size}_size_h" );
			$sizes[ $_size ][ 'crop' ]   = (bool) get_option( "{$_size}_crop" );
		} //in_array( $_size, array( 'thumbnail', 'medium', 'medium_large', 'large' ) )
		elseif ( isset( $_wp_additional_image_sizes[ $_size ] ) ) {
			$sizes[ $_size ] = array(
				 'width' => $_wp_additional_image_sizes[ $_size ][ 'width' ],
				'height' => $_wp_additional_image_sizes[ $_size ][ 'height' ],
				'crop' => $_wp_additional_image_sizes[ $_size ][ 'crop' ] 
			);
		} //isset( $_wp_additional_image_sizes[ $_size ] )
	} //get_intermediate_image_sizes() as $_size
	
	return $sizes;
}

function control_elementor_get_image_sizes_Arr() {
	$sizes = control_elementor_get_image_sizes();
  
	$arr = array();
	foreach ($sizes as $key => $value) {
	   $arr[$key] = $key.' - '. $value['width'].'X'.$value['height']. ' - '. (($value['crop'])? 'Cropped' : '');
	}
	
	return $arr;
}


function control_elementor_get_terms( $tax = 'category', $key = 'id' ) {
    $terms = array();

    if(!taxonomy_exists($tax)) return false;

    if ( $key === 'id' ) foreach ( (array) get_terms( $tax, array( 'hide_empty' => false ) ) as $term ) $terms[$term->term_id] = $term->name;
      elseif ( $key === 'slug' ) foreach ( (array) get_terms( $tax, array( 'hide_empty' => false ) ) as $term ) $terms[$term->slug] = $term->name;
        return $terms;
}