<?php

/**
 * This file represents an example of the code that themes would use to register
 * the required plugins.
 *
 * It is expected that theme authors would copy and paste this code into their
 * functions.php file, and amend to suit.
 *
 * @see http://tgmpluginactivation.com/configuration/ for detailed documentation.
 *
 * @package    TGM-Plugin-Activation
 * @subpackage Bohopeople plugins
 * @version    2.5.2
 * @author     Thomas Griffin, Gary Jones, Juliette Reinders Folmer
 * @copyright  Copyright (c) 2011, Thomas Griffin
 * @license    http://opensource.org/licenses/gpl-2.0.php GPL v2 or later
 * @link       https://github.com/TGMPA/TGM-Plugin-Activation
 */
/**
 * Include the TGM_Plugin_Activation class.
 */
require_once __DIR__ . '/class-tgm-plugin-activation.php';

add_action('tgmpa_register', 'investment_register_required_plugins');
/**
 * Register the required plugins for this theme.
 *
 * In this example, we register five plugins:
 * - one included with the TGMPA library
 * - two from an external source, one from an arbitrary source, one from a GitHub repository
 * - two from the .org repo, where one demonstrates the use of the `is_callable` argument
 *
 * The variable passed to tgmpa_register_plugins() should be an array of plugin
 * arrays.
 *
 * This function is hooked into tgmpa_init, which is fired within the
 * TGM_Plugin_Activation class constructor.
 */
if (!function_exists('investment_register_required_plugins')) :

    function investment_register_required_plugins() {
        /*
         * Array of plugin arrays. Required keys are name and slug.
         * If the source is NOT from the .org repo, then source is also required.
         */
        include_once ABSPATH . 'wp-admin/includes/plugin.php';
        $plugins = array(
            // This is an example of how to include a plugin bundled with a theme.
            array(
                'name' => __('Investment post types', 'investment'),
                'slug' => 'zinvestment_modules',
                'source' => __DIR__ . '/plugins/zinvestment_modules.zip',
                'required' => true,
                'version' => '1.6.8'
            ),
            array(
                'name' => __('Slider Revolution', 'investment'),
                'slug' => 'revslider',
                'source' => __DIR__ . '/plugins/revslider-6.5.8.zip',
                'required' => false,
                'version' => '6.5.8'
            ),
            array(
                'name' => __('Social Share', 'investment'),
                'slug' => 'easy-social-share-buttons3',
                'source' => __DIR__ . '/plugins/easy-social-share-buttons3-7.9.zip',
                'required' => false,
                'version' => '7.9'
            ),
            array(
                'name' => esc_attr__('Contact Form 7', 'investment'),
                'slug' => 'contact-form-7',
                'required' => true,
            ),
            array(
                'name' => esc_attr__('Breadcrumb NavXT', 'investment'),
                'slug' => 'breadcrumb-navxt',
                'required' => true,
            ),
            array(
                'name' => __('Post Types Order', 'investment'),
                'slug' => 'post-types-order',
                'required' => false,
            ),
            array(
                'name' => __('Email Subscription', 'investment'),
                'slug' => 'email-subscribers',
                'required' => false,
            ),
            array(
                'name' => __('One Click Demo Import', 'investment'),
                'slug' => 'one-click-demo-import',
                'required' => false,
            ),
            array(
                'name' => __('Control Elementor (Elementor page builder is required)', 'investment'),
                'slug' => 'control-elementor',
                'source' => __DIR__ . '/plugins/control-elementor.zip',
                'required' => false,
                'version' => '1.1.4',
            ),
            array(
                'name' => __('Elementor Page Builder', 'investment'),
                'slug' => 'elementor',
                'required' => false,
            ),
            array(
                'name' => __('WPBackery Page Builder', 'investment'),
                'slug' => 'js_composer',
                'source' => __DIR__ . '/plugins/js_composer-6.8.0.zip',
                'required' => false,
                'version' => '6.8.0',
            ),
            array(
                'name' => 'Envato market',
                'slug' => 'envato-market',
                'source' => 'https://goo.gl/pkJS33',
            ),
        );

   

        $config = array(
            'id' => 'tgmpa', // Unique ID for hashing notices for multiple instances of TGMPA.
            'default_path' => '', // Default absolute path to bundled plugins.
            'menu' => 'tgmpa-install-plugins', // Menu slug.
            'parent_slug' => 'themes.php', // Parent menu slug.
            'capability' => 'edit_theme_options', // Capability needed to view plugin install page, should be a capability associated with the parent menu used.
            'has_notices' => true, // Show admin notices or not.
            'dismissable' => true, // If false, a user cannot dismiss the nag message.
            'dismiss_msg' => '', // If 'dismissable' is false, this message will be output at top of nag.
            'is_automatic' => false, // Automatically activate plugins after installation or not.
            'message' => '<p><a class="" href="' . add_query_arg(['check' => 'plugin_status']) . '">Check ' . wp_get_theme()->get('Name') . '\'s custom plugins update availability</a></p>' // Message to output right before the plugins table.
        );
        $plugins = investment_fetch_plugins_update($plugins, false, $config);
        tgmpa($plugins, $config);
    }

endif;

add_filter('ocdi/register_plugins', 'investment_ocdi_register_plugins');

function investment_ocdi_register_plugins() {
    $plugins = [];
    if (isset($_GET['import']) && intval($_GET['import']) >= '8') {
        $plugins[] = array(
            'name' => __('Elementor', 'investment'),
            'slug' => 'elementor',
            'required' => true,
            'preselected' => true,
        );

        $plugins[] = array(
            'name' => __('Control Elementor', 'investment'),
            'slug' => 'control-elementor',
            'source' => __DIR__ . '/plugins/control-elementor.zip',
            'required' => true,
            'version' => '1.0.2',
            'preselected' => true,
        );
    } else {
        $plugins[] = array(
            'name' => __('WPBackery Page Builder', 'investment'),
            'slug' => 'js_composer',
            'source' => __DIR__ . '/plugins/js_composer-6.8.0.zip',
            'required' => true,
            'version' => '6.8.0',
            'preselected' => true,
        );
    }

    $plugins = array_merge($plugins, array(
        array(
            'name' => esc_attr__('Woocommerce', 'investment'),
            'slug' => 'woocommerce',
            'required' => false,
            'preselected' => true,
        ),
        array(
            'name' => esc_attr__('Woocommerce quick view', 'investment'),
            'slug' => 'yith-woocommerce-quick-view',
            'required' => false
        ),
        array(
            'name' => esc_attr__('Woocommerce wishlist', 'investment'),
            'slug' => 'yith-woocommerce-wishlist',
            'required' => false
        )
    ));

    return investment_fetch_plugins_update($plugins);
}

if (!function_exists('investment_fetch_plugins_update')) {

    /**
     * Helper function to fetch the Plugins array.
     *
     * @param array $plugins Whether or not to return a normalized array. 
     *
     * @return array
     *
     * @access public
     */
    function investment_fetch_plugins_update($plugins, $force_build = false, $config = []) {
        
        if(empty($config['menu'])) return $plugins;
        if(empty($_REQUEST['page'])) return $plugins;

        if($config['menu'] != $_REQUEST['page']) return $plugins;
        
        if (is_string($plugins) || is_bool($plugins))
            return $plugins;



        $update_data = wp_get_update_data();
        if ($update_data['counts']['plugins'] > 0) {
            $force_build = true;
        } elseif (!empty($_GET['check']) && ($_GET['check'] == 'plugin_status')) {
            $force_build = true;
        }

        // Plugins cache key.
        $plugins_cache_key = apply_filters('investment_plugins_cache_key', 'investment_plugins_cache');
        // Get the plugins from cache.
        $plugins_cache = apply_filters('investment_plugins_cache', get_transient($plugins_cache_key));

        if ($force_build || !is_array($plugins_cache) || empty($plugins_cache)) {

            $plugins_response = investment_get_plugin_api_response();

            // Continue if we got a valid response.
            if (200 === wp_remote_retrieve_response_code($plugins_response)) {
                $response_body = wp_remote_retrieve_body($plugins_response);

                if ($response_body) {
                    // JSON decode the response body and cache the result.
                    $plugins_data = json_decode(trim($response_body), true);
                    if (is_array($plugins_data) && isset($plugins_data['items'])) {


                        $plugins_tmp = [];
                        foreach ($plugins_data['items'] as $value) {
                            if (empty($value['source']))
                                continue;

                            $id = $value['slug'];
                            if ($id) {
                                $plugins_tmp[$id] = array(
                                    'version' => $value['Version'],
                                    'source' => $value['source'],
                                );
                            }
                        }

                        $plugins = investment_compare_plugins_version($plugins, $plugins_tmp);
                        set_theme_mod('investment_plugins', $plugins);
                        set_transient($plugins_cache_key, $plugins, WEEK_IN_SECONDS);
                    }
                }
            }
        } else {
            $plugins = $plugins_cache;
        }

        return $plugins;
    }

}

function investment_get_plugin_api_response() {
    // API url and key.
    $plugins_api_url = apply_filters('investment_plugins_api_url', 'https://www.jthemes.com/themes/tgmpa/');
    $plugins_api_key = apply_filters('investment_plugins_api_key', 'AIzaSyAY4CxRw0I0VvaABZcMcNqU-Zjuw7xjrW4');

    // API arguments.
    $plugins_fields = apply_filters('investment_plugins_fields', array('Slug', 'Version', 'Name'));
    $plugins_sort = apply_filters('investment_plugins_sort', 'alpha');
    // Initiate API request.
    $plugins_query_args = array(
        'key' => $plugins_api_key,
        'fields' => implode(',', $plugins_fields),
        'sort' => $plugins_sort,
    );

    // Build and make the request.
    $plugins_query = esc_url_raw(add_query_arg($plugins_query_args, $plugins_api_url));
    return wp_safe_remote_get(
            $plugins_query,
            array(
                'sslverify' => false,
                'timeout' => 15,
            )
    );
}

function investment_compare_plugins_version($plugins, $plugins_tmp) {
    // check custom plugins
    foreach ($plugins as $key => $plugin) {
        if (empty($plugin['source']))
            continue;
        if (!array_key_exists($plugin['slug'], $plugins_tmp))
            continue;

        $slug = $plugin['slug'];
        if (version_compare($plugins[$key]['version'], $plugins_tmp[$slug]['version'], '>='))
            continue;

        $plugins[$key]['version'] = $plugins_tmp[$slug]['version'];
        $plugins[$key]['source'] = $plugins_tmp[$slug]['source'];
    }
    return array_filter($plugins);
}
