<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Custom functions for WooCommerce
 *
 * @package   InsightFramework
 */
class Insight_Woo {

	const MINIMUM_PLUGIN_VERSION = '4.0.2';

	/**
	 * The constructor.
	 */
	public function __construct() {
		add_filter( 'woocommerce_add_to_cart_fragments', array( $this, 'woo_header_cart_fragment' ) );
		add_filter( 'subcategory_archive_thumbnail_size', array( $this, 'woo_subcategory_archive_thumbnail_size' ) );
		add_action( 'wp_footer', array( $this, 'woo_footer_actions' ) );

		// Hide default smart wishlist button
		add_filter( 'woosw_button_position_archive', function () {
			return '0';
		} );
		add_filter( 'woosw_button_position_single', function () {
			return '0';
		} );
		add_filter( 'woosw_color_default', function () {
			return Insight::PRIMARY_COLOR;
		} );

		// Check old version installed.
		if ( defined( 'WOOSCP_VERSION' ) || ( defined( 'WOOSC_VERSION' ) && version_compare( WOOSC_VERSION, self::MINIMUM_PLUGIN_VERSION, '<' ) ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_compare_plugin_version' ] );
		}

		// Hide default smart compare button
		add_filter( 'woosc_button_position_archive', '__return_false' );
		add_filter( 'woosc_button_position_single', '__return_false' );
		add_filter( 'woosc_bar_btn_color_default', function () {
			return Insight::PRIMARY_COLOR;
		} );

		// Hide default smart quick view button
		add_filter( 'woosq_button_position', function () {
			return '0';
		} );

		// Move section You may be interested in…
		remove_action( 'woocommerce_cart_collaterals', 'woocommerce_cross_sell_display' );
		add_action( 'woocommerce_after_cart_table', 'woocommerce_cross_sell_display' );

		// Ajax tab
		add_action( 'wp_ajax_insight_ajax_load_more_tab', array( $this, 'ajax_load_more_tab' ) );
		add_action( 'wp_ajax_nopriv_insight_ajax_load_more_tab', array( $this, 'ajax_load_more_tab' ) );

		// Ajax Product Grid
		add_action( 'wp_ajax_insight_product_grid', array( $this, 'insight_product_grid' ) );
		add_action( 'wp_ajax_nopriv_insight_product_grid', array( $this, 'insight_product_grid' ) );

		// Fix PHP warning from WC When an image use external url has no key: width + height
		add_filter( 'wp_get_attachment_metadata', array( $this, 'normalize_attachment_metadata' ), 9999, 2 );

		add_action( 'init', array( $this, 'remove_wc_register_blocks' ), 1 );
	}

	/**
	 * This WC action hook make Customize Preview loading freeze.
	 * Temp remove it in customize preview
	 *
	 * @return void
	 */
	public function remove_wc_register_blocks() {
		if ( is_customize_preview() ) {
			organik_remove_filters_for_anonymous_class( 'init', 'Automattic\WooCommerce\Blocks\BlockTypesController', 'register_blocks' );
		}
	}

	public static function header_wishlist() {
		$wishlist_html = '<a href="' . WPCleverWoosw::get_url() . '"><span class="wishlist-btn" data-count="' . WPCleverWoosw::get_count() . '"></span></a>';

		return $wishlist_html;
	}

	public function ajax_load_more_tab() {
		// Sanitize and validate incoming data
		$per_page = isset( $_POST['perpage'] ) ? intval( $_POST['perpage'] ) : 10;
		$page     = isset( $_POST['page'] ) ? intval( $_POST['page'] ) : 1;
		$order    = isset( $_POST['order'] ) ? sanitize_text_field( $_POST['order'] ) : '';
		$orderby  = isset( $_POST['orderby'] ) ? sanitize_text_field( $_POST['orderby'] ) : '';
		$cat      = isset( $_POST['cat'] ) ? sanitize_text_field( $_POST['cat'] ) : '';

		$args = array(
			'post_type'      => 'product',
			'posts_per_page' => $per_page,
			'paged'          => $page,
			'order'          => $order,
			'orderby'        => $orderby,
		);

		if ( $cat && $cat !== '*' ) {
			$args['tax_query'] = array(
				array(
					'taxonomy' => 'product_cat',
					'field'    => 'slug',
					'terms'    => $cat,
				),
			);
		}

		$query = new WP_Query( $args );

		ob_start(); // Start output buffering

		if ( $query->have_posts() ) {
			while ( $query->have_posts() ) {
				$query->the_post();
				wc_get_template_part( 'content', 'product' );
			}
			wp_reset_postdata();
			$content = ob_get_clean();
			wp_send_json_success( array( 'html' => $content ) );
		} else {
			ob_end_clean(); // Clean buffer if nothing was printed
			wp_send_json_error( array( 'message' => 'No more products.' ) );
		}

		wp_die();
	}


	public function insight_product_grid() {
		$args = array(
			'post_type'      => 'product',
			'posts_per_page' => isset( $_GET['posts_per_page'] ) ? intval( $_GET['posts_per_page'] ) : 10,
			'paged'          => isset( $_GET['paged'] ) ? intval( $_GET['paged'] ) : 1,
			'order'          => isset( $_GET['order'] ) ? sanitize_text_field( $_GET['order'] ) : '',
			'orderby'        => isset( $_GET['orderby'] ) ? sanitize_text_field( $_GET['orderby'] ) : '',
		);

		if ( ! empty( $_GET['categories'] ) ) {
			$args['tax_query'] = [
				'relation' => 'or',
				array(
					'taxonomy' => 'product_cat',
					'field'    => 'slug',
					'terms'    => explode( ',', $_GET['categories'] ),
				),
			];
		}

		$query = new WP_Query( $args );
		if ( $query->have_posts() ) {
			while ( $query->have_posts() ) {
				$query->the_post();
				wc_get_template_part( 'content', 'product' );
			}
			wp_reset_postdata();
		}

		wp_die();
	}

	function woo_header_cart_fragment( $fragments ) {
		ob_start();
		echo self::header_cart();
		$fragments['.mini-cart'] = ob_get_clean();

		return $fragments;
	}

	public static function header_cart( $mobile = false ) {
		if ( class_exists( 'WooCommerce' ) ) {
			$cart_url  = wc_get_cart_url();
			$qty       = WC()->cart->get_cart_contents_count();
			$total     = WC()->cart->get_cart_subtotal();
			$cart_html = '';

			if ( $mobile ) {
				$cart_html .= '<a href="' . esc_url( $cart_url ) . '">';
				$cart_html .= '<div class="mini-cart"><div class="mini-cart-icon" data-count="' . $qty . '"><i class="ion-bag"></i></div>';
				$cart_html .= '<div class="mini-cart-text">' . esc_html__( 'My Cart', 'tm-organik' ) . '<div class="mini-cart-total">' . $total . '</div></div></div>';
				$cart_html .= '</a>';
			} else {
				$cart_html .= '<div class="mini-cart"><div class="mini-cart-icon" data-count="' . $qty . '"><i class="ion-bag"></i></div>';
				$cart_html .= '<div class="mini-cart-text">' . esc_html__( 'My Cart', 'tm-organik' ) . '<div class="mini-cart-total">' . $total . '</div></div></div>';
			}

			return $cart_html;
		}
	}

	function woo_footer_actions() {
		if ( is_singular( 'product' ) ) {
			?>
			<script>
              jQuery(document).ready(function() {
                insightMarkProductViewed(<?php echo get_the_ID(); ?>);
              });
			</script>
			<?php
		}
	}

	function woo_subcategory_archive_thumbnail_size() {
		return 'full';
	}

	public function admin_notice_minimum_compare_plugin_version() {
		organik_notice_required_plugin_version( 'WPC Smart Compare for WooCommerce', '4.0.2' );
	}

	function normalize_attachment_metadata( $data, $attachment_id ) {
		if ( is_array( $data ) && ( isset( $data[0] ) && '' === $data[0] ) ) {
			$data = array();
		}

		/**
		 * Fix PHP warning from WC.
		 * When an image use external url has no key: width + height
		 */
		$data['width']  = $data['width'] ?? 0;
		$data['height'] = $data['height'] ?? 0;

		if ( is_array( $data ) && is_array( $data['sizes'] ) && ! empty( $data['sizes'] ) ) {
			foreach ( $data['sizes'] as $key => $_size ) {
				// If there's an exact match to an existing image size, short circuit.
				if ( empty( $_size ) || ! isset( $_size['width'] ) ) {
					unset( $data['sizes'][ $key ] );
				}
			}
		}

		return $data;
	}
}
