<?php
/**
 * Update Member Type
 *
 * @package     AutomatorWP\Integrations\BuddyPress\Triggers\Update_Member_Type
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyPress_Update_Member_Type extends AutomatorWP_Integration_Trigger {

    public $integration = 'buddypress';
    public $trigger = 'buddypress_update_member_type';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets assigned to a profile type', 'automatorwp-pro' ),
            'select_option'     => __( 'User gets assigned to a <strong>profile type</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Profile type. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets assigned to %1$s %2$s time(s)', 'automatorwp-pro' ), '{member_type}', '{times}' ),
            /* translators: %1$s: Profile type. */
            'log_label'         => sprintf( __( 'User gets assigned to %1$s', 'automatorwp-pro' ), '{member_type}' ),
            'action'            => 'bp_set_member_type',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'member_type' => array(
                    'from' => 'member_type',
                    'default' => 'any',
                    'fields' => array(
                        'member_type' => array(
                            'name' => __( 'Profile Type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => 'automatorwp_buddypress_member_types_options_cb',
                            'option_none' => true,
                            'default' => 'any'
                        ),
                    ),
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $user_id
     * @param string    $member_type
     * @param bool      $append
     */
    public function listener( $user_id, $member_type, $append ) {

        // Bail if assigned to a blank member type
        if( $member_type === '' ) {
            return;
        }

        // Trigger the member type update
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'member_type'   => $member_type,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if member type is not received
        if( ! isset( $event['member_type'] ) ) {
            return false;
        }

        // Don't deserve if group doesn't match with the trigger option
        if( $trigger_options['member_type'] !== 'any' && $trigger_options['member_type'] !== $event['member_type'] ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_BuddyPress_Update_Member_Type();