<?php
/**
 * Functions
 *
 * @package     AutomatorWP\Integrations\Facebook\Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Get the Facebook API version
 *
 * @since 1.0.0
 *
 * @return string
 */
function automatorwp_facebook_api_version() {

    return 'v15.0';

}

/**
 * Get the Facebook pages
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_facebook_get_pages() {

    $fb_pages = array();

    $transient = get_transient( 'automatorwp_facebook_pages' );

    if ( $transient !== false ) {
        return $transient;
    }

    $params = automatorwp_facebook_get_request_parameters( );
    $api_version = automatorwp_facebook_api_version();

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return $fb_pages;
    }
 
    $auth = get_option( 'automatorwp_facebook_auth' );

    $args = array(
        'body' => array(
            'access_token'  => $auth['access_token'],
            'limit'         => 100,
            'offset'        =>0,
        )
    );

    // Get the Facebook pages
    $response = wp_remote_get('https://graph.facebook.com/'. $api_version .'/me/accounts', $args);
    $response = json_decode( wp_remote_retrieve_body( $response ), true );
    
    $pages = $response['data'];

    if( isset( $pages ) && is_array( $pages ) ) {

        foreach( $pages as $page ) {

                $fb_pages[] = array(
                    'id' => $page['id'],
                    'name' => $page['name'],
                );

        } 
    }

    if( count( $fb_pages ) ) {
        // Set a transient for 1 day with the Facebook pages
        set_transient( 'automatorwp_facebook_pages', $fb_pages, 24 * HOUR_IN_SECONDS );
    }

    return $fb_pages;
}

/**
 * Options callback for select2 fields assigned to pages
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_facebook_options_cb_page( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any page', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $page_id ) {

            // Skip option none
            if( $page_id === $none_value ) {
                continue;
            }

            $options[$page_id] = automatorwp_facebook_get_page_title( $page_id );
        }
    }

    return $options;

}

/**
 * Get the Facebook page title
 *
 * @since 1.0.0
 *
 * @param int $page_id
 *
 * @return string|null
 */
function automatorwp_facebook_get_page_title( $page_id ) {

    // Empty title if no ID provided
    if( absint( $page_id ) === 0 ) {
        return '';
    }

    $pages = automatorwp_facebook_get_pages();
    $page_name = '';

    foreach( $pages as $page ) {

        if( $page['id'] === $page_id ) {
            $page_name = $page['name'];
            break;
        }
    }

    return $page_name;

}

/**
 * Get the Facebook groups
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_facebook_get_groups() {

    $fb_groups = array();

    $transient = get_transient( 'automatorwp_facebook_groups' );

    if ( $transient !== false ) {
        return $transient;
    }

    $params = automatorwp_facebook_get_request_parameters( );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return $fb_groups;
    }
 
    $api_version = automatorwp_facebook_api_version();
    $auth = get_option( 'automatorwp_facebook_auth' );
    $access_token = $auth['access_token'];
    $application_id = automatorwp_facebook_get_option( 'application_id', '' );

    // Get the Facebook groups
    $response = wp_remote_get('https://graph.facebook.com/' . $api_version . '/me/groups?access_token=' . $access_token );

    $response = json_decode( wp_remote_retrieve_body( $response ), true );
    $groups = $response['data'];

    if( isset( $groups ) && is_array( $groups ) ) {

        foreach( $groups as $group ) {

                $fb_groups[] = array(
                    'id' => $group['id'],
                    'name' => $group['name'],
                );

        } 
    }

    if( count( $fb_groups ) ) {
        // Set a transient for 1 day with the Facebook groups
        set_transient( 'automatorwp_facebook_groups', $fb_groups, 24 * HOUR_IN_SECONDS );
    }

    return $fb_groups;
}

/**
 * Options callback for select2 fields assigned to groups
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_facebook_options_cb_group( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any group', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $group_id ) {

            // Skip option none
            if( $group_id === $none_value ) {
                continue;
            }

            $options[$group_id] = automatorwp_facebook_get_group_title( $group_id );
        }
    }

    return $options;

}

/**
 * Get the Facebook group title
 *
 * @since 1.0.0
 *
 * @param int $group_id
 *
 * @return string|null
 */
function automatorwp_facebook_get_group_title( $group_id ) {

    // Empty title if no ID provided
    if( absint( $group_id ) === 0 ) {
        return '';
    }

    $groups = automatorwp_facebook_get_groups();
    $group_name = '';

    foreach( $groups as $group ) {

        if( $group['id'] === $group_id ) {
            $group_name = $group['name'];
            break;
        }
    }

    return $group_name;

}


/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @param string $platform
 *
 * @return array|false
 */
function automatorwp_facebook_get_request_parameters( ) {

    $auth = get_option( 'automatorwp_facebook_auth' );
    
    if( ! is_array( $auth ) ) {
        return false;
    }

    return array(
        'user-agent'  => 'AutomatorWP; ' . home_url(),
        'timeout'     => 120,
        'httpversion' => '1.1',
        'headers'     => array(
            'Authorization' => 'Bearer ' . $auth['access_token'],
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json'
        )
    );
}

