<?php
/**
 * Remove HTML
 *
 * @package     AutomatorWP\Integrations\Formatter\Actions\Remove_HTML
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly

if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Formatter_Remove_HTML extends AutomatorWP_Integration_Action {

    public $integration = 'formatter';
    public $action = 'formatter_remove_html';
    public $formatted = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Remove html in a string', 'automatorwp-pro' ),
            'select_option'     => __( 'Remove <strong>html</strong> in a <strong>string</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: String. %2$s: Format */
            'edit_label'        => sprintf( __( 'Remove html in a %1$s', 'automatorwp-pro' ), '{string}' ),
            'log_label'         => sprintf( __( 'Remove html in a %1$s', 'automatorwp-pro' ), '{string}' ),
            'options'           => array(
                'string' => array(
                    'default' => __( 'string', 'automatorwp-pro' ),
                    'fields' => array(
                        'string' => array(
                            'name' => __( 'String:', 'automatorwp-pro' ),
                            'desc' => __( 'String to remove the HTML.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                     ) ),
            ),
            'tags'  => automatorwp_formatter_get_actions_string_tags()
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $string = $action_options['string'];

        // Bail if not data
        if ( empty( $string ) ) {
            $this->result = __( 'Please, insert a string to be formatted', 'automatorwp-pro' );
            return;
        }

        // Format string      
        $this->formatted = strip_tags( $string );
        
        $this->result = sprintf( __( '%s', 'automatorwp-pro' ), $this->formatted );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;
        $log_meta['formatted'] = ( isset( $this->formatted ) ? $this->formatted : '' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Formatter_Remove_HTML();