<?php
/**
 * Achievement Type Expires
 *
 * @package     AutomatorWP\Integrations\GamiPress\Triggers\Achievement_Type_Expires
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_Achievement_Type_Expires extends AutomatorWP_Integration_Trigger {

    public $integration = 'gamipress';
    public $trigger = 'gamipress_achievement_type_expires';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User achievement of a type expires', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>achievement of a type</strong> expires', 'automatorwp-pro' ),
            /* translators: %1$s: Post type. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User %1$s expires %2$s time(s)', 'automatorwp-pro' ), '{post_type}', '{times}' ),
            /* translators: %1$s: Post type. */
            'log_label'         => sprintf( __( 'User %1$s expires', 'automatorwp-pro' ), '{post_type}' ),
            'action'            => 'gamipress_expirations_earning_expired',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 4,
            'options'           => array(
                'post_type' => array(
                    'from' => 'post_type',
                    'fields' => array(
                        'post_type' => array(
                            'name' => __( 'Achievement type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-selector',
                            'options_cb' => array( $this, 'options_cb_post_types' ),
                            'default' => 'any'
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option()
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Options callback for achievement type options
     *
     * @since 1.0.0
     *
     * @param stdClass $field
     *
     * @return array
     */
    public function options_cb_post_types( $field ) {

        $options = array(
            'any' => __( 'an achievement of any type', 'automatorwp-pro' ),
        );

        foreach( gamipress_get_achievement_types() as $post_type => $data ) {
            $options[$post_type] = sprintf( __( 'a %s', 'automatorwp-pro' ), strtolower( $data['singular_name'] ) );
        }

        return $options;

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $user_id            The user ID
     * @param int       $post_id            The post ID
     * @param int       $user_earning_id    The user earning ID
     * @param stdClass  $user_earning       The user earning object
     */
    public function listener( $user_id, $post_id, $user_earning_id, $user_earning ) {

        $post = get_post( $post_id );

        // Bail if not post instanced
        if( ! $post instanceof WP_Post ) {
            return;
        }

        // Bail if post type is not an achievement
        if( ! in_array( $post->post_type, gamipress_get_achievement_types_slugs() ) ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger' => $this->trigger,
            'user_id' => $user_id,
            'post_id' => $post->ID,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        $post = get_post( absint( $event['post_id'] ) );

        // Don't deserve if post doesn't exists
        if( ! $post ) {
            return false;
        }

        $post_type = $trigger_options['post_type'];

        // Don't deserve if post doesn't match with the trigger option
        if( $post_type !== 'any' && $post->post_type !== $post_type ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_GamiPress_Achievement_Type_Expires();