<?php
/**
 * Tags
 *
 * @package     AutomatorWP\Generator\Tags
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

// GLOBAL TAGS

/**
 * Generator tags
 *
 * @since 1.0.0
 *
 * @param array $tags The tags
 *
 * @return array
 */
function automatorwp_generator_get_tags( $tags ) {

    $tags['generator'] = array(
        'label' => 'Generator',
        'tags'  => array(),
        'icon'  => AUTOMATORWP_GENERATOR_URL . 'assets/generator.svg',
    );

    $tags['generator']['tags']['generate_hash( VALUE : ALGORITHM )'] = array(
        'label'     => __( 'Hash', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => __( 'Generate a hash from the given value, replace "VALUE" by the value of your choice. Replace "ALGORITHM" by the algorithm of your choice. By default MD5.', 'automatorwp-pro' ),
    );

    $tags['generator']['tags']['generate_nonce( ACTION )'] = array(
        'label'     => __( 'Nonce', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => __( 'Generates a WordPress nonce, replace "ACTION" by the action name of your choice.', 'automatorwp-pro' ),
    );

    $tags['generator']['tags']['generate_number( MIN : MAX )'] = array(
        'label'     => __( 'Random number', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => __( 'Generates a random number, replace "MIN" and "MAX" by the values of your choice.', 'automatorwp-pro' ),
    );

    $tags['generator']['tags']['generate_string( LENGTH : [ CHARS ] )'] = array(
        'label'     => __( 'Random string', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => __( 'Generates a random string, replace "LENGTH" by the length of your choice. By default 12.  Replace "CHARS" by a single or comma-separated list of characters allowed. Accepts LOWERCASE, UPPERCASE, NUMBER, SPECIAL and ALL. By default ALL.', 'automatorwp-pro' ),
    );

    $tags['generator']['tags']['generate_email( [ NAMES ] : [ DOMAINS ] : [ EXTS ] )'] = array(
        'label'     => __( 'Generate email', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => __( 'Generates a email from a group of options, replace "NAMES", "DOMAINS" and "EXTS" by a single or comma-separated list of usernames, domains and extensions.', 'automatorwp-pro' ),
    );

    $tags['generator']['tags']['generate_random_email( NAME_LENGTH : [NAME_CHARS] : DOMAIN_LENGTH : [DOMAIN_CHARS] )'] = array(
        'label'     => __( 'Random email', 'automatorwp-pro' ),
        'type'      => 'text',
        'preview'   => __( 'Generates a random email, replace "NAME_LENGTH" and "DOMAIN_LENGTH" by the lengths of your choice. By default 12.  Replace "NAME_CHARS" and "DOMAIN_CHARS" by a comma-separated list of characters allowed. Accepts LOWERCASE, UPPERCASE, NUMBER, SPECIAL and ALL. By default ALL.', 'automatorwp-pro' ),
    );

    return $tags;

}
add_filter( 'automatorwp_get_tags', 'automatorwp_generator_get_tags' );

/**
 * Calculator tags names
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_generator_get_tags_names() {

    $tags = automatorwp_generator_get_tags( array() );
    $tags_names = array();

    foreach( $tags['generator']['tags'] as $tag => $args ) {
        $tags_names[] = explode( '(', $tag )[0];
    }

    return $tags_names;

}

/**
 * Skip tags replacements
 *
 * @since 1.0.0
 *
 * @param bool      $skip           If tag should get skipped, by default false
 * @param string    $tag_name       The tag name (without "{}")
 * @param int       $automation_id  The automation ID
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 *
 * @return bool
 */
function automatorwp_generator_skip_tag_replacement( $skip, $tag_name, $automation_id, $user_id, $content ) {

    foreach( automatorwp_generator_get_tags_names() as $tag ) {
        if( automatorwp_starts_with( $tag_name, $tag . '(' ) ) {
            return true;
        }
    }

    return $skip;

}
add_filter( 'automatorwp_skip_tag_replacement', 'automatorwp_generator_skip_tag_replacement', 10, 5 );

/**
 * Parse tags
 *
 * @since 1.0.0
 *
 * @param string    $parsed_content     Content parsed
 * @param array     $replacements       Automation replacements
 * @param int       $automation_id      The automation ID
 * @param int       $user_id            The user ID
 * @param string    $content            The content to parse
 *
 * @return string
 */
function automatorwp_generator_post_parse_automation_tags( $parsed_content, $replacements, $automation_id, $user_id, $content ) {

    $replacements = array();

    if( empty( $parsed_content ) ) {
        return $parsed_content;
    }

    // Get the functions
    preg_match_all( "/\{\s*(.*?)\s*\(\s*(.*)\s*\)\s*\}/", $parsed_content, $matches );

    if( is_array( $matches ) && isset( $matches[1] ) ) {

        foreach( $matches[1] as $i => $function ) {

            // Skip if not is our function
            if( ! in_array( $function, automatorwp_generator_get_tags_names() ) ) {
                continue;
            }

            $parsed_params = automatorwp_parse_automation_tags( $automation_id, $user_id, $matches[2][$i] );
            $params = automatorwp_get_tag_parameters_to_array( $function, $parsed_params );
            $value = $params[0];
            $all_allowed_characters = array(
                'lowercase',
                'uppercase',
                'number',
                'special',
            );

            switch ( $function ) {
                case 'generate_hash':
                    $algorithm = isset( $params[1] ) ? strtolower( $params[1] ) : 'md5';

                    $all_algorithms = hash_algos();
                    $index = array_search( $algorithm, $all_algorithms );

                    if( $index === false ) {
                        $index = 2; // md5 index
                    }

                    $value = hash( $all_algorithms[$index], $value );
                    break;
                case 'generate_nonce':
                    $value = wp_create_nonce( $value );
                    break;
                case 'generate_number':
                    $min = isset( $params[0] ) ? absint( $params[0] ) : 0;
                    $max = isset( $params[1] ) ? absint( $params[1] ) : 999999;

                    $value = mt_rand( $min , $max );
                    break;
                case 'generate_string':
                    $length = absint( $params[0] );
                    $chars = isset( $params[1] ) ? $params[1] : 'ALL';
                    $allowed_characters = array();

                    // Sanitize length
                    if( $length === 0 ) $length = 12;

                    if( is_array( $chars ) ) {
                        foreach( $chars as $char ) {
                            $allowed_char = strtolower( trim( $char ) );

                            if( $allowed_char === 'ALL' ) {
                                $chars = 'ALL';
                                break;
                            }

                            $allowed_characters[] = $allowed_char;
                        }
                    }

                    if( $chars === 'ALL' ) {
                        $allowed_characters = $all_allowed_characters;
                    }

                    $value = automatorwp_generator_random_string( $length, $allowed_characters );
                    break;
                case 'generate_email':
                    $name = '';
                    $domain = '';
                    $ext = '';

                    $names = $params[0];
                    $domains = isset( $params[1] ) ? $params[1] : array();
                    $exts = isset( $params[2] ) ? $params[2] : array();

                    // Ensure values as arrays
                    if( ! is_array( $names ) ) $names = array( $names );
                    if( ! is_array( $domains ) ) $domains = array( $domains );
                    if( ! is_array( $exts ) ) $exts = array( $exts );

                    // Pick a random one
                    if ( count( $names ) > 1 ) $name = $names[array_rand( $names, 1 )];
                    if ( count( $domains ) > 1 ) $domain = $domains[array_rand( $domains, 1 )];
                    if ( count( $domains ) > 1 ) $ext = $exts[array_rand( $domains, 1 )];

                    // Generate a random one if empty
                    if( empty( $name ) ) {
                        $name = automatorwp_generator_random_string( mt_rand( 4 , 64 ), array( 'lowercase', 'number' ) );
                    }

                    if( empty( $domain ) ) {
                        $domain = automatorwp_generator_random_string( mt_rand( 4 , 64 ), array( 'lowercase', 'number' ) );
                    }

                    if( empty( $ext ) ) {
                        $ext = automatorwp_generator_random_string( mt_rand( 2 , 4 ), array( 'lowercase' ) );
                    }

                    $value = str_replace( ' ', '', $name . '@' . $domain . '.' . $ext );
                    break;
                case 'generate_random_email': // NAME_LENGTH, [NAME_CHARS], DOMAIN_LENGTH, [DOMAIN_CHARS]
                    $name_length = absint( $params[0] );
                    $name_chars = isset( $params[1] ) ? $params[1] : array();
                    $domain_length = isset( $params[2] ) ? absint( $params[2] ) : 0;
                    $domain_chars = isset( $params[3] ) ? $params[3] : array();
                    $name_allowed_characters = array();
                    $domain_allowed_characters = array();

                    // Sanitize lengths
                    if( $name_length === 0 ) $name_length = 12;
                    if( $domain_length === 0 ) $domain_length = 12;

                    if( is_array( $name_chars ) ) {
                        foreach( $name_chars as $char ) {
                            $allowed_char = strtolower( trim( $char ) );

                            if( $allowed_char === 'ALL' ) {
                                $name_chars = 'ALL';
                                break;
                            }

                            $name_allowed_characters[] = $allowed_char;
                        }
                    }

                    if( $name_chars === 'ALL' ) {
                        $name_allowed_characters = $all_allowed_characters;
                    }

                    if( is_array( $domain_chars ) ) {
                        foreach( $domain_chars as $char ) {
                            $allowed_char = strtolower( trim( $char ) );

                            if( $allowed_char === 'ALL' ) {
                                $domain_chars = 'ALL';
                                break;
                            }

                            $domain_allowed_characters[] = $allowed_char;
                        }
                    }

                    if( $domain_chars === 'ALL' ) {
                        $domain_allowed_characters = $all_allowed_characters;
                    }

                    $name = automatorwp_generator_random_string( $name_length, $name_allowed_characters );
                    $domain = automatorwp_generator_random_string( $domain_length, $domain_allowed_characters );
                    $ext = automatorwp_generator_random_string( mt_rand( 2 , 4 ), array( 'lowercase' ) );

                    $value = str_replace( ' ', '', $name . '@' . $domain . '.' . $ext );
                    break;
            }

            $replacements[$matches[0][$i]] = $value;
        }

    }

    if( count( $replacements ) > 0 ) {
        $tags = array_keys( $replacements );

        // Replace all tags by their replacements
        $parsed_content = str_replace( $tags, $replacements, $parsed_content );
    }

    return $parsed_content;
}
add_filter( 'automatorwp_post_parse_automation_tags', 'automatorwp_generator_post_parse_automation_tags', 10, 5 );

// ACTION TAGS

/**
 * Random number tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_generator_get_actions_random_number_tags() {

    return array(
        'random_number' => array(
            'label'     => __( 'Random number', 'automatorwp-pro' ),
            'type'      => 'integer',
            'preview'   => '123',
        ),
    );

}

/**
 * Custom action random number tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $action         The action object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last action log object
 *
 * @return string
 */
function automatorwp_generator_get_action_random_number_tag_replacement( $replacement, $tag_name, $action, $user_id, $content, $log ) {


    $action_args = automatorwp_get_action( $action->type );

    // Skip if action is not from this integration
    if( $action_args['integration'] !== 'generator' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'random_number':
            $replacement = automatorwp_get_log_meta( $log->id, 'random_number', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_action_tag_replacement', 'automatorwp_generator_get_action_random_number_tag_replacement', 10, 6 );

/**
 * Random string tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_generator_get_actions_random_string_tags() {

    return array(
        'random_string' => array(
            'label'     => __( 'Random string', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'random_string',
        ),
    );

}

/**
 * Custom action random string tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $action         The action object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last action log object
 *
 * @return string
 */
function automatorwp_generator_get_action_random_string_tag_replacement( $replacement, $tag_name, $action, $user_id, $content, $log ) {


    $action_args = automatorwp_get_action( $action->type );

    // Skip if action is not from this integration
    if( $action_args['integration'] !== 'generator' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'random_string':
            $replacement = automatorwp_get_log_meta( $log->id, 'random_string', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_action_tag_replacement', 'automatorwp_generator_get_action_random_string_tag_replacement', 10, 6 );

/**
 * Random email tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_generator_get_actions_random_email_tags() {

    return array(
        'random_email' => array(
            'label'     => __( 'Random email', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'contact@automatorwp.com',
        ),
    );

}

/**
 * Custom action random email tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $action         The action object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last action log object
 *
 * @return string
 */
function automatorwp_generator_get_action_random_email_tag_replacement( $replacement, $tag_name, $action, $user_id, $content, $log ) {


    $action_args = automatorwp_get_action( $action->type );

    // Skip if action is not from this integration
    if( $action_args['integration'] !== 'generator' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'random_email':
            $replacement = automatorwp_get_log_meta( $log->id, 'random_email', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_action_tag_replacement', 'automatorwp_generator_get_action_random_email_tag_replacement', 10, 6 );

/**
 * Random hash tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_generator_get_actions_random_hash_tags() {

    return array(
        'random_hash' => array(
            'label'     => __( 'Random hash', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'random_hash',
        ),
    );

}

/**
 * Custom action random hash tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $action         The action object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last action log object
 *
 * @return string
 */
function automatorwp_generator_get_action_random_hash_tag_replacement( $replacement, $tag_name, $action, $user_id, $content, $log ) {


    $action_args = automatorwp_get_action( $action->type );

    // Skip if action is not from this integration
    if( $action_args['integration'] !== 'generator' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'random_hash':
            $replacement = automatorwp_get_log_meta( $log->id, 'random_hash', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_action_tag_replacement', 'automatorwp_generator_get_action_random_hash_tag_replacement', 10, 6 );

/**
 * Random nonce tags
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_generator_get_actions_random_nonce_tags() {

    return array(
        'random_nonce' => array(
            'label'     => __( 'Random nonce', 'automatorwp-pro' ),
            'type'      => 'text',
            'preview'   => 'random_nonce',
        ),
    );

}

/**
 * Custom action random nonce tag replacement
 *
 * @since 1.0.0
 *
 * @param string    $replacement    The tag replacement
 * @param string    $tag_name       The tag name (without "{}")
 * @param stdClass  $action         The action object
 * @param int       $user_id        The user ID
 * @param string    $content        The content to parse
 * @param stdClass  $log            The last action log object
 *
 * @return string
 */
function automatorwp_generator_get_action_random_nonce_tag_replacement( $replacement, $tag_name, $action, $user_id, $content, $log ) {


    $action_args = automatorwp_get_action( $action->type );

    // Skip if action is not from this integration
    if( $action_args['integration'] !== 'generator' ) {
        return $replacement;
    }

    switch( $tag_name ) {
        case 'random_nonce':
            $replacement = automatorwp_get_log_meta( $log->id, 'random_nonce', true );
            break;
    }

    return $replacement;

}
add_filter( 'automatorwp_get_action_tag_replacement', 'automatorwp_generator_get_action_random_nonce_tag_replacement', 10, 6 );