<?php
/**
 * Mark Lesson Assignments
 *
 * @package     AutomatorWP\Integrations\LearnDash\Actions\Mark_Lesson_Assignments
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_Mark_Lesson_Assignments extends AutomatorWP_Integration_Action {

    public $integration = 'learndash';
    public $action = 'learndash_mark_lesson_assignments';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Mark user assignments from lesson as approved or not approved', 'automatorwp-pro' ),
            'select_option'     => __( 'Mark user assignments from lesson as <strong>approved or not approved</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Operation (approved or not approved). */
            'edit_label'        => sprintf( __( 'Mark user assignments from %1$s as %2$s', 'automatorwp-pro' ), '{post}', '{operation}' ),
            /* translators: %1$s: Post title. %2$s: Operation (approved or not approved). */
            'log_label'         => sprintf( __( 'Mark user assignments from %1$s as %2$s', 'automatorwp-pro' ), '{post}', '{operation}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Lesson:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'all lessons', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Lesson ID', 'automatorwp-pro' ),
                    'post_type'         => 'sfwd-lessons',
                ) ),
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'approve'      => __( 'approved', 'automatorwp-pro' ),
                                'not_approve'  => __( 'not approved', 'automatorwp-pro' ),
                            ),
                            'default' => 'approve'
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $lesson_id = $action_options['post'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'approve';
        }

        // Check specific lesson
        if( $lesson_id !== 'any' ) {

            // Get user assignments for this lesson
            $query = new WP_Query( array(
                'fields'        => 'ids',
                'post_type'     => 'sfwd-assignment',
                'post_author'   => $user_id,
                'meta_query'    => array(
                    array(
                        'key'   => 'lesson_id',
                        'value' => $lesson_id,
                    ),
                ),
            ) );

        } else {

            // Get all user assignments
            $query = new WP_Query( array(
                'fields'        => 'ids',
                'post_type'     => 'sfwd-assignment',
                'post_author'   => $user_id,
            ) );

        }

        $assignments_ids = $query->get_posts();

        // Loop assignments
        foreach ( $assignments_ids as $assignment_id ) {

            switch ( $operation ) {
                case 'approve':
                    // Mark lesson assignment as approved
                    learndash_assignment_mark_approved( $assignment_id );

                    /**
                     * Fires after assignment is approved
                     *
                     * @since 2.2.0
                     *
                     * @param int $assignment_id Assignment ID.
                     */
                    do_action( 'learndash_assignment_approved', $assignment_id );
                    break;
                case 'not_approve':
                    // Mark lesson assignment as not approved
                    update_post_meta( $assignment_id, 'approval_status', 0 );
                    break;
            }

        }



    }

}

new AutomatorWP_LearnDash_Mark_Lesson_Assignments();