<?php
/**
 * Add subscriber
 *
 * @package     AutomatorWP\Integrations\MailerLite\Actions\Add_Contact
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_MailerLite_Add_Subscriber extends AutomatorWP_Integration_Action {

    public $integration = 'mailerlite';
    public $action = 'mailerlite_add_subscriber';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add/Update subscriber to MailerLite', 'automatorwp-pro' ),
            'select_option'     => __( 'Add/Update <strong>subscriber</strong> to MailerLite', 'automatorwp-pro' ),
            /* translators: %1$s: Subscriber. */
            'edit_label'        => sprintf( __( 'Add/Update %1$s to MailerLite', 'automatorwp-pro' ), '{subscriber}' ),
            /* translators: %1$s: Subscriber. */
            'log_label'         => sprintf( __( 'Add/Update %1$s to MailerLite', 'automatorwp-pro' ), '{subscriber}' ),
            'options'           => array(
                'subscriber' => array(
                    'from' => 'email',
                    'default' => __( 'subscriber', 'automatorwp-pro' ),
                    'fields' => array(
                        'email' => array(
                            'name' => __( 'Email:', 'automatorwp-pro' ),
                            'desc' => __( 'The subscriber email address.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'first_name' => array(
                            'name' => __( 'First Name:', 'automatorwp-pro' ),
                            'desc' => __( 'The subscriber\'s first name.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'last_name' => array(
                            'name' => __( 'Last Name:', 'automatorwp-pro' ),
                            'desc' => __( 'The subscriber\'s last name.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $subscriber_data = array(
            'email'         => $action_options['email'],
            'first_name'    => $action_options['first_name'],
            'last_name'     => $action_options['last_name'],
        );

        $this->result = '';
        
        // Bail if user email is empty
        if ( empty( $subscriber_data['email'] ) ) {
            $this->result = __( 'Email subscriber field is empty.', 'automatorwp-pro' );
            return;
        }

        // Bail if MailerLite not configured
        if( ! automatorwp_mailerlite_get_api() ) {
            $this->result = __( 'MailerLite integration not configured in AutomatorWP settings.', 'automatorwp-pro' );
            return;
        }

        $api = automatorwp_mailerlite_get_api();

    if( ! $api ) {
        return $options;
    }

    $test = wp_remote_get( 'https://api.mailerlite.com/api/v2/groups', array(
        'headers' => array(
            'Authorization' => 'Bearer ' . $api['token'],
            'Accept' => 'application/json',
            'Content-Type'  => 'application/json'
        ),
    ) );
    error_log('--action--');
        error_log(print_r($test,1));
    $test = json_decode( wp_remote_retrieve_body( $test ), true  );
        

        $response = automatorwp_mailerlite_add_subscriber( $subscriber_data );

        if ( $response === 201 ) {

            $this->result = sprintf( __( '%s added to MailerLite', 'automatorwp-pro' ), $subscriber_data['email'] );

        }

        if ( $response === 200 ) {

            $this->result = sprintf( __( '%s updated in MailerLite', 'automatorwp-pro' ), $subscriber_data['email'] );
            
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        // Warn user if the authorization has not been setup from settings
        if( ! automatorwp_mailerlite_get_api() ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">MailerLite settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-mailerlite'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/mailerlite/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_MailerLite_Add_Subscriber();