<?php
/**
 * Update User Field
 *
 * @package     AutomatorWP\Integrations\Meta_Box\Actions\Update_User_Field
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Meta_Box_Update_User_Field extends AutomatorWP_Integration_Action {

    public $integration = 'meta_box';
    public $action = 'meta_box_update_user_field';

    /**
     * The action result
     *
     * @since 1.0.0
     *
     * @var array $result
     */
    public $result = array();

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Update user field with a value', 'automatorwp-pro' ),
            'select_option'     => __( 'Update <strong>user field</strong> with a value', 'automatorwp-pro' ),
            /* translators: %1$s: User. %2$s: Field name. %1$s: Field value. */
            'edit_label'        => sprintf( __( 'Update %1$s %2$s with %3$s', 'automatorwp-pro' ), '{user}', '{field_name}', '{field_value}' ),
            /* translators: %1$s: User. %2$s: Field name. %1$s: Field value. */
            'log_label'         => sprintf( __( 'Update %1$s %2$s with %3$s', 'automatorwp-pro' ), '{user}', '{field_name}', '{field_value}' ),
            'options'           => array(
                'user' => array(
                    'default' => __( 'user', 'automatorwp-pro' ),
                    'fields' => array(
                        'user_id' => array(
                            'name' => __( 'User ID:', 'automatorwp-pro' ),
                            'desc' => __( 'The user\'s ID to update.', 'automatorwp-pro' )
                                . ' ' . __( 'Leave empty to use the ID of the user that completes the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    )
                ),

                'field_name' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'field_name',
                    'option_default'    => __( 'Select a field', 'automatorwp-pro' ),
                    'name'              => __( 'Field:', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_meta_box_get_user_fields',
                    'options_cb'        => 'automatorwp_meta_box_options_cb_user_fields',
                    'default'           => ''
                ) ),

                'field_value' => array(
                    'from' => 'field_value',
                    'default' => __( 'any value', 'automatorwp-pro' ),
                    'fields' => array(
                        'field_value' => array(
                            'name' => __( 'Value:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        global $wpdb;

        // Shorthand
        $field_name = $action_options['field_name'];
        $field_value = $action_options['field_value'];
        $this->result = array();

        // User ID
        if( empty( $action_options['user_id'] ) ) {
            $action_options['user_id'] = $user_id;
        }

        $user_id = absint( $action_options['user_id'] );
        $user = get_userdata( $user_id );

        // Bail if could not find the user
        if( ! $user ) {
            $this->result[] = sprintf( __( 'User not found by the ID %1$s.', 'automatorwp-pro' ), $user_id );
            return;
        }

        // Update user meta
        update_user_meta( $user_id, $field_name, $field_value );
        
        $this->result[] = __( 'User field updated successfully.', 'automatorwp-pro' );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store user ID
        $log_meta['user_id'] = $user_id;

        // Store user field
        $log_meta['field_name'] = $action_options['field_name'];

        // Store user value field
        $log_meta['field_value'] = $action_options['field_value'];

        // Store result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['user_title'] = array(
            'name' => __( 'User Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about the user updated.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['user_id'] = array(
            'name' => __( 'User ID:', 'automatorwp-pro' ),
            'desc' => __( 'The user\'s ID updated.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['field_name'] = array(
            'name' => __( 'User Field:', 'automatorwp-pro' ),
            'desc' => __( 'The updated field.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['field_value'] = array(
            'name' => __( 'User Field Value:', 'automatorwp-pro' ),
            'desc' => __( 'The updated field value.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Meta_Box_Update_User_Field();