<?php
/**
 * Add Attachment
 *
 * @package     AutomatorWP\Integrations\WordPress\Triggers\Add_Attachment
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WordPress_Add_Attachment extends AutomatorWP_Integration_Trigger {

    /**
     * Initialize the trigger
     *
     * @since 1.0.0
     */
    public function __construct( $integration ) {

        $this->integration = $integration;
        $this->trigger = $integration . '_add_attachment';

        parent::__construct();

    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User uploads a multimedia file', 'automatorwp-pro' ),
            'select_option'     => __( 'User uploads a <strong>multimedia file</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Multimedia type (image, video or audio). %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User uploads %1$s file %2$s time(s)', 'automatorwp-pro' ), '{mime_type}', '{times}' ),
            /* translators: %1$s: Multimedia type (image, video or audio). */
            'log_label'         => sprintf( __( 'User uploads %1$s file', 'automatorwp-pro' ), '{mime_type}' ),
            'action'            => 'add_attachment',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'mime_type' => array(
                    'from' => 'mime_type',
                    'fields' => array(
                        'mime_type' => array(
                            'name' => __( 'Multimedia type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'any'       => __( 'any multimedia', 'automatorwp-pro' ),
                                'image'    => __( 'an image', 'automatorwp-pro' ),
                                'video'    => __( 'a video', 'automatorwp-pro' ),
                                'audio'    => __( 'an audio', 'automatorwp-pro' ),
                            ),
                            'default' => 'any'
                        ),
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $attachment_id The attachment ID
     */
    public function listener( $attachment_id ) {

        $attachment = get_post( $attachment_id );

        // Return if attachment not exists
        if( ! $attachment ) {
            return;
        }

        $user_id = absint( $attachment->post_author );

        // Trigger likes
        automatorwp_trigger_event( array(
            'trigger'           => $this->trigger,
            'user_id'           => $user_id,
            'post_id'           => $attachment_id,
            'mime_type'         => $attachment->post_mime_type,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if mime type is not received
        if( ! isset( $event['mime_type'] ) ) {
            return false;
        }

        if( $trigger_options['mime_type'] !== 'any' ) {
            $mime_type = explode( '/',  $event['mime_type'] )[0];

            // Don't deserve if mime type doesn't match with the trigger option
            if( $trigger_options['mime_type'] !== $mime_type ) {
                return false;
            }
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WordPress_Add_Attachment( 'wordpress' );
new AutomatorWP_WordPress_Add_Attachment( 'multimedia_content' );