<?php
/**
 * Functions
 *
 * @package     AutomatorWP\Webhooks\Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Get all registered request methods
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_webhooks_get_request_methods() {

    return apply_filters( 'automatorwp_webhooks_get_request_methods', array(
        'POST'  => 'POST',
        'GET'   => 'GET',
        'PUT'   => 'PUT',
    ) );

}

/**
 * Get all registered request formats
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_webhooks_get_request_formats() {

    return apply_filters( 'automatorwp_webhooks_get_request_formats', array(
        'FORM'  => 'FORM',
        'JSON'  => 'JSON',
    ) );

}

/**
 * Set the default URL value
 *
 * @since 1.0.0
 *
 * @param array         $field_args Current field args
 * @param CMB2_Field    $field      Current field object
 *
 * @return string
 */
function automatorwp_webhooks_url_default_cb( $field_args, $field ) {

    $slug = ct_get_object_meta( $field->object_id, 'slug', true );

    return get_rest_url() . 'automatorwp/webhooks/' . $slug;

}

/**
 * Generates a random slug (used for receive triggers)
 *
 * @since 1.0.0
 *
 * @return string
 */
function automatorwp_webhooks_generate_slug() {

    global $wpdb;

    $random_url = strtolower( wp_generate_password( 8, false ) );

    $ct_table = ct_setup_table( 'automatorwp_triggers' );

    $exists = $wpdb->get_var(
        "SELECT tm.id 
        FROM {$ct_table->meta->db->table_name} AS tm 
        WHERE tm.meta_key = 'url_key' AND tm.meta_value = '{$random_url}'"
    );

    ct_reset_setup_table();

    // If generated random url exists, try to generate it again recursively
    if( $exists ) {
        $random_url = automatorwp_webhooks_generate_slug();
    }

    return $random_url;

}

/**
 * Get a receive webhook trigger by their slug
 *
 * @since 1.0.0
 *
 * @param string $slug
 *
 * @return stdClass
 */
function automatorwp_webhooks_get_trigger_by_slug( $slug ) {

    global $wpdb;

    $ct_table = ct_setup_table( 'automatorwp_triggers' );

    $trigger_types =  automatorwp_webhooks_get_integrations_trigger_types();

    $trigger = $wpdb->get_row( $wpdb->prepare(
        "SELECT t.* 
        FROM {$ct_table->db->table_name} AS t 
        LEFT JOIN {$ct_table->meta->db->table_name} AS tm ON ( t.id = tm.id AND tm.meta_key = 'slug' )
        WHERE t.type IN ( '" . implode( "', '", $trigger_types ) . "' )
        AND tm.meta_value = %s
        LIMIT 1",
        $slug
    ) );

    ct_reset_setup_table();

    return $trigger;

}

/**
 * Get all receive webhook triggers
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_webhooks_get_receive_triggers() {

    global $wpdb;

    $cache = automatorwp_get_cache( 'automatorwp_webhooks_receive_triggers', false );

    // If result already cached, return it
    if( is_array( $cache ) ) {
        return $cache;
    }

    $trigger_types =  automatorwp_webhooks_get_integrations_trigger_types();

    $ct_table = ct_setup_table( 'automatorwp_triggers' );

    $triggers = $wpdb->get_results( "SELECT * FROM {$ct_table->db->table_name} AS t WHERE t.type IN ( '" . implode( "', '", $trigger_types ) . "' )" );

    ct_reset_setup_table();

    automatorwp_save_cache( 'automatorwp_webhooks_receive_triggers', $triggers );

    return $triggers;

}

/**
 * Fires once a trigger has been saved.
 *
 * @since 1.0.0
 *
 * @param int       $object_id  The object ID.
 * @param stdClass  $object     The object.
 * @param bool      $update     Whether this is an existing post being updated or not.
 */
function automatorwp_webhooks_on_insert_trigger( $object_id, $object, $update ) {

    global $ct_table;

    // Bail if not is a trigger
    if( $ct_table->name !== 'automatorwp_triggers' ) {
        return;
    }

    // Bail if not is a receive webhook trigger
    if( ! in_array( $object->type, automatorwp_webhooks_get_integrations_trigger_types() ) ) {
        return;
    }

    // Only run this function on insert
    if( $update ) {
        return;
    }

    // Reset receive triggers cache
    automatorwp_delete_cache( 'automatorwp_webhooks_receive_triggers' );

}
add_action( 'ct_insert_object', 'automatorwp_webhooks_on_insert_trigger', 10, 3 );

/**
 * Fires immediately before a trigger is deleted from the database.
 *
 * @since 1.0.0
 *
 * @param int       $object_id  The object ID.
 */
function automatorwp_webhooks_on_delete_trigger( $object_id ) {

    global $ct_table;

    // Bail if not is a trigger
    if( $ct_table->name !== 'automatorwp_triggers' ) {
        return;
    }

    $object = ct_get_object( $object_id );

    // Bail if trigger doesn't exists
    if( ! $object ) {
        return;
    }

    // Bail if not is a receive webhook trigger
    if( ! in_array( $object->type, automatorwp_webhooks_get_integrations_trigger_types() ) ) {
        return;
    }

    // Reset receive triggers cache
    automatorwp_delete_cache( 'automatorwp_webhooks_receive_triggers' );

}
add_action( 'delete_object', 'automatorwp_webhooks_on_delete_trigger', 10 );

/**
 * Helper function to merge arrays recursively
 * The PHP array_merge_recursive() function causes this issue:
 * array( 'key/0/sub_key_1' => '', 'key/0/sub_key_2' => '' )
 * array( 'key' => array(
 *  array( 'sub_key_1' => ''  ),
 *  array( 'sub_key_2' => ''  )
 * ) )
 * JSON:
 * "key":[
 *  {"sub_key_1":""},
 *  {"sub_key_2":""}
 * ]
 *
 * This functions merge them correctly as:
 * array( 'key/0/sub_key_1' => '', 'key/0/sub_key_2' => '' )
 * array( 'key' => array(
 *  'sub_key_1' => '',
 *  'sub_key_2' => ''
 * ) )
 * JSON: "key":[{
 *  "sub_key_1":"",
 *  "sub_key_2":""
 * }]
 *
 * @since 1.0.0
 *
 * @param array $array1  Original array.
 * @param array $array2  Array to merge.
 *
 * @return array
 */
function automatorwp_webhooks_array_merge_recursive( array &$array1, array &$array2 ) {
    $merged = $array1;

    foreach ( $array2 as $key => &$value ) {
        if ( is_array( $value ) && isset( $merged[$key] ) && is_array( $merged[$key] ) ) {
            $merged[$key] = automatorwp_webhooks_array_merge_recursive ( $merged[$key], $value );
        } else {
            $merged[$key] = $value;
        }
    }

    return $merged;
}

/**
 * Recursively create the hierarchy for nested fields.
 *
 * @param array $response The response.
 * @param string $prefix The prefix for the keys.
 * @return array
 */
function automatorwp_webhooks_action_nested_fields( $response, $prefix = '' ) {
    $result = [];

    foreach ( $response as $key => $value ) {

        $new_key = $prefix ? $prefix . '/' . $key : $key;
        
        if ( is_array( $value ) ) {
            $result = array_merge( $result, automatorwp_webhooks_action_nested_fields( $value, $new_key ) );
        } else {
            $result[$new_key] = $value;
        }

    }

    return $result;
}