<?php
/**
 * Anonymous Receive Webhook
 *
 * @package     AutomatorWP\Integrations\AutomatorWP\Triggers\Anonymous_Receive_Webhook
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Webhooks_Anonymous_Receive_Webhook extends AutomatorWP_Integration_Trigger {

    /**
     * The integration label
     *
     * @since 1.0.0
     *
     * @var string $label
     */
    public $label = '';

    /**
     * Initialize the trigger
     *
     * @since 1.0.0
     */
    public function __construct( $integration, $label ) {

        $this->integration = $integration;
        $this->trigger = $integration . '_anonymous_receive_webhook';
        $this->label = $label;

        parent::__construct();

    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'anonymous'         => true,
            /* translators: %1$s: Integration. */
            'label'             => sprintf( __( 'Receive data from %1$s', 'automatorwp-pro' ), $this->label ),
            /* translators: %1$s: Integration. */
            'select_option'     => sprintf( __( 'Receive data from %1$s', 'automatorwp-pro' ), "<strong>{$this->label}</strong>" ),
            /* translators: %1$s: Integration. */
            'edit_label'        => sprintf( __( 'Receive data from %1$s', 'automatorwp-pro' ), '{webhook}' ),
            /* translators: %1$s: Integration. */
            'log_label'         => sprintf( __( 'Receive data from %1$s', 'automatorwp-pro' ), '{webhook}' ),
            'action'            => 'automatorwp_webhook_received',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 5,
            'options'           => array(
                'webhook' => array(
                    'default' => $this->label,
                    'fields' => array(
                        'url' => array(
                            'name' => __( 'URL:', 'automatorwp-pro' ),
                            'desc' => __( 'Send your webhook to this URL. Supports GET, POST and PUT methods.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'classes' => 'automatorwp-webhooks-url',
                            'attributes' => array(
                                'disabled' => 'disabled'
                            ),
                            'default_cb' => 'automatorwp_webhooks_url_default_cb',
                        ),
                        'fields' => array(
                            'name' => __( 'Fields:', 'automatorwp-pro' ),
                            'type' => 'group',
                            'classes' => 'automatorwp-fields-table automatorwp-webhooks-trigger-fields',
                            'options'     => array(
                                'add_button'        => __( 'Add pair', 'automatorwp-pro' ),
                                'remove_button'     => '<span class="dashicons dashicons-no-alt"></span>',
                            ),
                            'fields' => array(
                                'required' => array(
                                    'name' => __( 'Required:', 'automatorwp-pro' ),
                                    'type' => 'checkbox',
                                    'classes' => 'cmb2-switch'
                                ),
                                'key' => array(
                                    'name' => __( 'Key:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                                'type' => array(
                                    'name' => __( 'Type:', 'automatorwp-pro' ),
                                    'type' => 'select',
                                    'options' => array(
                                        'text'          => __( 'Text', 'automatorwp-pro' ),
                                        'integer'       => __( 'Integer', 'automatorwp-pro' ),
                                        'float'         => __( 'Float', 'automatorwp-pro' ),
                                        'array'         => __( 'Array', 'automatorwp-pro' ),
                                    ),
                                    'default' => 'text'
                                ),
                                'required_value' => array(
                                    'name' => __( 'Required value:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                            ),
                        ),
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // On insert trigger
        add_action( 'ct_insert_object', array( $this, 'on_insert_trigger' ), 10, 3 );

        // Get samples button
        add_filter( 'automatorwp_automation_ui_before_option_form', array( $this, 'before_option_form' ), 10, 5 );

        // Samples dialogs
        add_action( 'admin_footer', array( $this, 'admin_footer' ) );

        // Register dynamic trigger tags
        add_filter( 'automatorwp_trigger_tags', array( $this, 'dynamic_tags' ), 10, 2 );

        // Dynamic trigger tags replacements
        add_filter( 'automatorwp_trigger_tags_replacements', array( $this, 'dynamic_tags_replacements' ), 10, 4 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        // Exclude slug and URL meta to being cloned or exported
        add_filter( "automatorwp_clone_trigger_{$this->trigger}_meta_slug_excluded", '__return_true' );
        add_filter( "automatorwp_clone_trigger_{$this->trigger}_meta_url_excluded", '__return_true' );
        add_filter( "automatorwp_export_url_trigger_{$this->trigger}_meta_slug_excluded", '__return_true' );
        add_filter( "automatorwp_export_url_trigger_{$this->trigger}_meta_url_excluded", '__return_true' );

        parent::hooks();
    }

    /**
     * Fires once a trigger has been saved.
     *
     * @since 1.0.0
     *
     * @param int       $object_id  The object ID.
     * @param stdClass  $object     The object.
     * @param bool      $update     Whether this is an existing post being updated or not.
     */
    public function on_insert_trigger( $object_id, $object, $update ) {

        global $ct_table;

        // Bail if not is a trigger
        if( $ct_table->name !== 'automatorwp_triggers' ) {
            return;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return;
        }

        // Only run this function on insert
        if( $update ) {
            return;
        }

        $slug = automatorwp_webhooks_generate_slug();

        ct_update_object_meta( $object_id, 'slug', $slug );

        // Reset receive triggers cache
        delete_option( 'automatorwp_cache_automatorwp_webhooks_receive_triggers' );

    }

    /**
     * Before option form custom content
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     * @param string    $option     The option key
     * @param array     $args       The option arguments
     */
    public function before_option_form( $object, $item_type, $option, $args ) {

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return;
        }

        ?>
        <div class="automatorwp-webhook-get-samples-response"></div>
        <button type="button" class="button automatorwp-webhooks-get-samples"><?php _e( 'Get Samples', 'automatorwp-pro' ); ?></button>
        <div class="clear" style="margin-bottom: 15px;"></div>
        <?php

    }

    /**
     * Custom content on admin footer
     */
    public function admin_footer() {
        ?>
        <div class="automatorwp-webhooks-samples-dialog" style="display: none">
            <h2><?php _e( 'Here is the data we\'ve collected', 'automatorwp-pro' ); ?></h2>
            <div class="automatorwp-webhooks-samples-options"></div>
            <button type="button" class="button button-primary automatorwp-webhooks-samples-confirm"><?php _e( 'Use these fields', 'automatorwp-pro' ); ?></button>
            <button type="button" class="button automatorwp-webhooks-samples-cancel"><?php _e( 'Cancel', 'automatorwp-pro' ); ?></button>
        </div>
        <?php
    }

    /**
     * Register trigger dynamic tags
     *
     * @since 1.0.0
     *
     * @param array     $tags       The trigger tags
     * @param stdClass  $trigger    The trigger object
     * @return mixed
     */
    public function dynamic_tags( $tags, $trigger ) {

        // Bail if trigger type don't match this trigger
        if( $trigger->type !== $this->trigger ) {
            return $tags;
        }

        $trigger_options = automatorwp_get_trigger_stored_options( $trigger->id );

        if( isset( $trigger_options['fields'] ) && is_array( $trigger_options['fields'] ) ) {

            // Add the stored fields as tags
            foreach( $trigger_options['fields'] as $field ) {

                // Skip fields that are not well configured
                if( ! isset( $field['type'] ) || ! isset( $field['key'] ) ) {
                    continue;
                }

                // Setup the field preview based on the field type
                switch( $field['type'] ) {
                    case 'integer':
                    case 'user_id':
                        $field_preview = '1';
                        break;
                    case 'user_email':
                        $field_preview = 'contact@automatorwp.com';
                        break;
                    case 'float':
                        $field_preview = '1.00';
                        break;
                    case 'text':
                    default:
                        $field_preview = __( 'Text', 'automatorwp-pro' );
                        break;
                }

                $tag_type = $field['type'];

                if( $tag_type === 'user_id' ) {
                    $tag_type = 'integer';
                } else if( $tag_type === 'user_email' ) {
                    $tag_type = 'text';
                }

                $tags[$trigger->id . ':' . $field['key']] = array(
                    'label' => $field['key'],
                    'type' => $tag_type,
                    'preview' => $field_preview
                );

            }

        }

        return $tags;

    }

    /**
     * Setup trigger dynamic tags replacements
     *
     * @since 1.0.0
     *
     * @param array     $replacements   The trigger replacements
     * @param stdClass  $trigger        The trigger object
     * @param int       $user_id        The user ID
     * @param stdClass  $log            The last trigger log object
     *
     * @return array
     */
    function dynamic_tags_replacements( $replacements, $trigger, $user_id, $log ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $replacements;
        }

        // Get fields stored on last log
        $fields = ct_get_object_meta( $log->id, 'fields', true );

        if( is_array( $fields ) ) {

            // Append the fields values as replacements
            foreach( $fields as $key => $value ) {
                $replacements[$key] = $value;
            }

        }

        return $replacements;

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param string            $slug       The slug matched
     * @param stdClass          $trigger    The trigger object
     * @param int               $user_id    The user ID
     * @param array             $fields     Fields received
     * @param WP_REST_Request   $data       WordPress Rest API request object
     */
    public function listener( $slug, $trigger, $user_id, $fields, $data ) {

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'slug'          => $slug,
            'trigger_id'    => $trigger->id,
            'fields'        => $fields,
        ) );

    }

    /**
     * Anonymous deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if anonymous deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                         True if anonymous deserves trigger, false otherwise
     */
    public function anonymous_deserves_trigger( $deserves_trigger, $trigger, $event, $trigger_options, $automation ) {

        // Don't deserve if slug and trigger are not received
        if( ! isset( $event['slug'] ) && ! isset( $event['trigger_id'] ) ) {
            return false;
        }

        // Get the trigger slug
        ct_setup_table( 'automatorwp_triggers' );
        $slug = ct_get_object_meta( $trigger->id, 'slug', true );
        ct_reset_setup_table();

        // Don't deserve if slug doesn't match with the trigger slug
        if( $slug !== $event['slug'] ) {
            return false;
        }

        // Don't deserve if trigger doesn't match with the trigger from event
        if( absint( $trigger->id ) !== absint( $event['trigger_id'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['url'] = get_rest_url() . 'automatorwp/webhooks/' . $event['slug'];
        $log_meta['fields'] = $event['fields'];

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an trigger
        if( $log->type !== 'trigger' ) {
            return $log_fields;
        }

        // Bail if trigger type don't match this trigger
        if( $object->type !== $this->trigger ) {
            return $log_fields;
        }

        $log_fields['webhook_info'] = array(
            'name' => __( 'Webhook Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about request received.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['url'] = array(
            'name' => __( 'URL:', 'automatorwp-pro' ),
            'desc' => __( 'URL of the request.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['fields'] = array(
            'name' => __( 'Fields:', 'automatorwp-pro' ),
            'desc' => __( 'Data received on the request.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}