<?php

/**
 * Booking Scheduled
 *
 * @package     AutomatorWP\Integrations\BookingCalendar\Triggers\Booking_Scheduled
 * @author      AutomatorWP <contact@automatorwp.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BookingCalendar_Booking_Scheduled extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_booking_calendar';
    public $trigger = 'wp_booking_calendar_booking_scheduled';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User schedules a <strong>booking</strong>', 'automatorwp-pro' ),
            'select_option'     => __( 'User schedules a <strong>booking</strong>', 'automatorwp-pro' ),
            'edit_label'        => sprintf( __( 'User schedules a booking %1$s time(s)', 'automatorwp-pro' ), '{times}' ),
            'log_label'         => __( 'User schedules a booking', 'automatorwp-pro' ),
            'action'            => 'wpbc_booking_approved',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'times' => automatorwp_utilities_times_option()
            ),
            'tags' => array_merge(
                automatorwp_wp_booking_calendar_get_booking_tags(),
                automatorwp_utilities_times_tag()
            ),
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param array $params Booking parameters passed to the listener
     * @param array $action_result Result of the action
     */
    public function listener( $booking_id ) {

        $user_id = get_current_user_id();

        // Login is required
        if ( $user_id === 0 ) {
            return;
        }

        // Triggers the event
        automatorwp_trigger_event( array(
            'trigger'     => $this->trigger,
            'user_id'     => $user_id,
            'booking_id'  => $booking_id,
        ) );
        
    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['booking_id'] = ( isset( $event['booking_id'] ) ? $event['booking_id'] : '' );

        return $log_meta;

    }

}

new AutomatorWP_BookingCalendar_Booking_Scheduled();
