<?php
/**
 * User List
 *
 * @package     AutomatorWP\Integrations\ActiveMember360\Actions\User_List
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_ActiveMember360_User_List extends AutomatorWP_Integration_Action {

    public $integration = 'activemember360';
    public $action = 'activemember360_user_list';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Subscribe or unsubscribe user from list', 'automatorwp-pro' ),
            'select_option'     => __( 'Subscribe or unsubscribe user from <strong>list</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (subscribe or unsubscribe). %2$s: List name. */
            'edit_label'        => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{list}' ),
            /* translators: %1$s: Operation (subscribe or unsubscribe). %2$s: List name. */
            'log_label'         => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{list}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'subscribe'     => __( 'Subscribe', 'automatorwp-pro' ),
                                'unsubscribe'   => __( 'Unsubscribe', 'automatorwp-pro' ),
                            ),
                            'default' => 'subscribe'
                        ),
                    )
                ),
                'list' => array(
                    'from' => 'list',
                    'fields' => array(
                        'list' => array(
                            'name' => __( 'List:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'classes' => 'automatorwp-selector',
                            'options_cb' => array( $this, 'lists_options_cb' ),
                            'default' => 'any'
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * Get lists options
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function lists_options_cb() {

        $options = array(
            'any' => __( 'all lists', 'automatorwp-pro' ),
        );

        // Get site lists
        $lists = apply_filters( 'mbr/site_lists/get', NULL );

        if( ! empty( $lists ) ) {
            foreach( $lists as $list_id => $list_name ) {
                $options[$list_id] = $list_name;
            }
        }

        return $options;

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'unsubscribe' ) {
            return sprintf( __( '%1$s user from %2$s', 'automatorwp-pro' ), '{operation}', '{list}' );
        }

        return $label;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $list_id = $action_options['list'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'subscribe';
        }

        $user = get_user_by('id', $user_id);
        $user_email = $user->user_email;

        // Check if user is a contact in ActiveCampaign
        $user_am360 = mbr_get_contact_by_email( $user_email );

        if ( empty( $user_am360 ) )  {
            return;
        }

        $lists = array();

        // Check specific list
        if( $list_id !== 'any' ) {

            $lists = array( $list_id );

        } else if( $list_id === 'any' ) {

            // Get all lists
            $all_lists = apply_filters( 'mbr/site_lists/get', NULL );

            if( ! empty( $all_lists ) ) {
                $lists = array_keys( $all_lists );
            }
        }

        // Setup the data to sync
        $data = array(
            'contact_data' => array(
                'email' => $user_am360['email']
            )
        );

        switch ( $operation ) {
            case 'subscribe':

                // Subscribe user to lists
                $data['subscribe_lists'] = $lists;

                break;
            case 'unsubscribe':

                // Unsubscribe user from lists
                $data['unsubscribe_lists'] = $lists;

                break;
        }

        // Sync user account
        apply_filters( 'mbr/api/sync_ctla', $data );

    }

}

new AutomatorWP_ActiveMember360_User_List();