<?php
/**
 * Reject Private Group Type
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Triggers\Reject_Private_Group_Type
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_Reject_Private_Group_Type extends AutomatorWP_Integration_Trigger {

    public $integration = 'buddyboss';
    public $trigger = 'buddyboss_reject_private_group_type';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets rejected in a private group of a type', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>gets rejected</strong> in a private group <strong>of a type</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Group. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets rejected in a private group of %1$s %2$s time(s)', 'automatorwp-pro' ), '{group_type}', '{times}' ),
            /* translators: %1$s: Group. */
            'log_label'         => sprintf( __( 'User gets rejected in a private group of %1$s', 'automatorwp-pro' ), '{group_type}' ),
            'action'            => array(
                'groups_membership_rejected',
            ),
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'group_type' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'group_type',
                    'name'              => __( 'Type:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any type', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddyboss_get_group_types',
                    'options_cb'        => 'automatorwp_buddyboss_options_cb_group_type',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_buddyboss_get_group_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $user_id
     * @param int $group_id
     * @param bool $value
     */
    public function listener( $user_id, $group_id, $value ) {

        // Trigger the get accepted in a private group
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'group_id'      => $group_id,
            'post_id'       => $group_id,
            'group_type'    => bp_groups_get_group_type( $group_id ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if group is not received
        if( ! isset( $event['group_id'] ) && ! isset( $event['group_type'] ) ) {
            return false;
        }

        // Don't deserve if group type doesn't match with the trigger option
        if( $trigger_options['group_type'] !== 'any' && $event['group_type'] !== $trigger_options['group_type'] ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['group_id'] = ( isset( $event['group_id'] ) ? $event['group_id'] : '' );
        $log_meta['group_type'] = ( isset( $event['group_type'] ) ? $event['group_type'] : '' );


        return $log_meta;

    }

}

new AutomatorWP_BuddyBoss_Reject_Private_Group_Type();