<?php
/**
 * User Achievement
 *
 * @package     AutomatorWP\Integrations\GamiPress\Actions\User_Achievement
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_User_Achievement extends AutomatorWP_Integration_Action {

    public $integration = 'gamipress';
    public $action = 'gamipress_user_achievement';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Award or revoke achievement to user', 'automatorwp-pro' ),
            'select_option'     => __( 'Award or revoke <strong>achievement</strong> to user', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (award or revoke). %2$s: Post title. %3$s: User. */
            'edit_label'        => sprintf( __( '%1$s the achievement %2$s to %3$s', 'automatorwp-pro' ), '{operation}', '{post}', '{user}' ),
            /* translators: %1$s: Operation (award or revoke). %2$s: Post title. %3$s: User. */
            'log_label'         => sprintf( __( '%1$s the achievement %2$s to %3$s', 'automatorwp-pro' ), '{operation}', '{post}', '{user}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'award'     => __( 'Award', 'automatorwp-pro' ),
                                'revoke'    => __( 'Revoke', 'automatorwp-pro' ),
                            ),
                            'default' => 'award'
                        ),
                    )
                ),
                'post' => automatorwp_gamipress_utilities_post_option( array(
                    'name' => __( 'Achievement:', 'automatorwp-pro' ),
                    'option_default' => __( 'Select an achievement', 'automatorwp-pro' ),
                    'option_none' => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Achievement ID', 'automatorwp-pro' ),
                    'post_type_cb' => 'gamipress_get_achievement_types_slugs'
                ) ),
                'user' => array(
                    'from' => 'user',
                    'default' => __( 'user', 'automatorwp-pro' ),
                    'fields' => array(
                        'user' => array(
                            'name' => __( 'User ID:', 'automatorwp-pro' ),
                            'desc' => __( 'User ID that will receive this achievement. Leave blank to award the achievement to the user that completes the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $post_id = absint( $action_options['post'] );
        $user_id_to_award = absint( $action_options['user'] );

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'award';
        }

        if( $user_id_to_award === 0 ) {
            $user_id_to_award = $user_id;
        }

        $user = get_userdata( $user_id_to_award );

        // Bail if user does not exists
        if( ! $user ) {
            return;
        }

        $post = get_post( $post_id );

        // Bail if post doesn't exists
        if( ! $post ) {
            return;
        }

        // Bail if post is not an achievement
        if( ! in_array( $post->post_type, gamipress_get_achievement_types_slugs() ) ) {
            return;
        }

        switch ( $operation ) {
            case 'award':
                // Award the achievement
                gamipress_award_achievement_to_user( $post_id, $user_id_to_award );
                break;
            case 'revoke':
                // Revoke the achievement
                gamipress_revoke_achievement_to_user( $post_id, $user_id_to_award );
                break;
        }

    }

}

new AutomatorWP_GamiPress_User_Achievement();