<?php
/**
 * User Points
 *
 * @package     AutomatorWP\Integrations\GamiPress\Filters\User_Points
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_User_Points_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'gamipress';
    public $filter = 'gamipress_user_points';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User points balance', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>points balance</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Points. %3$s: Points Type. */
            'edit_label'        => sprintf( __( '%1$s %2$s %3$s', 'automatorwp-pro' ), '{condition}', '{points}', '{points_type}'  ),
            /* translators: %1$s: Condition. %2$s: Points. %3$s: Points Type. */
            'log_label'         => sprintf( __( '%1$s %2$s %3$s', 'automatorwp-pro' ), '{condition}', '{points}', '{points_type}' ),
            'options'           => array(
                'condition' => automatorwp_utilities_number_condition_option(),
                'points' => array(
                    'from' => 'points',
                    'fields' => array(
                        'points' => array(
                            'name' => __( 'Points amount:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '1'
                        )
                    )
                ),
                'points_type' => array(
                    'from' => 'points_type',
                    'default' => __( 'points type', 'automatorwp-pro' ),
                    'fields' => array(
                        'points_type' => array(
                            'name' => __( 'Points Type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'option_none' => false,
                            'options_cb' => 'automatorwp_gamipress_points_types_options_cb'
                        )
                    )
                ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Shorthand
        $condition = $filter_options['condition'];
        $points = absint( $filter_options['points'] );
        $points_type = $filter_options['points_type'];

        // Bail if post is not a points type
        if( ! gamipress_get_points_type( $points_type ) ) {
            $this->result = __( 'Filter not passed. Points type option is not a registered points type.', 'automatorwp-pro' );
            return false;
        }

        $user_points = gamipress_get_user_points( $user_id, $points_type );

        // Don't deserve if user balance does not matches with this condition
        if( ! automatorwp_number_condition_matches( $user_points, $points, $condition ) ) {
            $this->result = sprintf( __( 'Filter not passed. User balance of %1$s %2$s does not meets the condition %3$s %4$s %5$s.', 'automatorwp-pro' ),
                $user_points,
                gamipress_get_points_type_plural( $points_type ),
                automatorwp_utilities_get_condition_label( $condition ),
                $points,
                gamipress_get_points_type_plural( $points_type )
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User balance of %1$s %2$s meets the condition %3$s %4$s %5$s.', 'automatorwp-pro' ),
            $user_points,
            gamipress_get_points_type_plural( $points_type ),
            automatorwp_utilities_get_condition_label( $condition ),
            $points,
            gamipress_get_points_type_plural( $points_type )
        );

        return $deserves_filter;

    }

}

new AutomatorWP_GamiPress_User_Points_Filter();