<?php
/**
 * Add Event
 *
 * @package     AutomatorWP\Integrations\Google_Calendar\Actions\Add_Event
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;


class AutomatorWP_Google_Calendar_Add_Event extends AutomatorWP_Integration_Action {
    
    public $integration = 'google_calendar';
    public $action = 'google_calendar_add_event';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        //$default_datetime = strtotime( date( 'Y-m-d H:i:s', current_time( 'timestamp' ) ) );

        automatorwp_register_action( $this->action, array(
            'integration' => $this->integration,
            'label' => __( 'Add an event to a calendar', 'automatorwp-pro' ),
            'select_option' => __( 'Add <strong>an event</strong> to a calendar', 'automatorwp-pro' ),
            /* translators: %1$s: Event. %2$s: Calendar. */
            'edit_label' => sprintf( __( 'Add %1$s to %2$s', 'automatorwp-pro' ), '{event}', '{calendar}' ),
            /* translators: %1$s: Event. %2$s: Calendar. */
            'log_label' => sprintf( __( 'Add %1$s to %2$s', 'automatorwp-pro' ), '{event}', '{calendar}' ),
            'options' => array(
                'calendar' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'calendar',
                    'option_default'    => __( 'calendar', 'automatorwp-pro' ),
                    'placeholder'       => __( 'Select a calendar', 'automatorwp-pro' ),
                    'name'              => __( 'Calendar:', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_google_calendar_get_calendars',
                    'options_cb'        => 'automatorwp_google_calendar_options_cb_calendars',
                    'default'           => ''
                ) ),
                        
                'event' => array(
                    'from' => 'event',
                    'default' => __( 'event', 'automatorwp-pro' ),
                    'fields' => array(
                        'event' => array(
                            'name' => __( 'Event:', 'automatorwp-pro' ),
                            'desc' => __( 'The event name.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'description' => array(
                            'name' => __( 'Description:', 'automatorwp-pro' ),
                            'desc' => __( 'The event description.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'start_datetime' => array(
                            'name' => __( 'Start date and time:', 'automatorwp-pro' ),
                            'desc' => __( 'The event start date (YYYY-MM-DD).', 'automatorwp-pro' ),
                            'type' => 'text_datetime_timestamp',
                            'date_format' =>  automatorwp_get_date_format( array( 'Y-m-d', 'm/d/Y' ) ),
                            'time_format' =>  automatorwp_get_time_format(),
                            'required'  => true,
                            'default' => '',
                        ),
                        'end_datetime' => array(
                            'name' => __( 'End date and time:', 'automatorwp-pro' ),
                            'desc' => __( 'The event end date (YYYY-MM-DD).', 'automatorwp-pro' ),
                            'type' => 'text_datetime_timestamp',
                            'date_format' =>  automatorwp_get_date_format( array( 'Y-m-d', 'm/d/Y' ) ),
                            'time_format' =>  automatorwp_get_time_format(),
                            'required'  => true,
                            'default' => '',
                        ),
                        'attendees' => array(
                            'name' => __( 'Attendees:', 'automatorwp-pro' ),
                            'desc' => __( 'Add the attendees separated by commas (user_1@example.com, user_2@example.com).', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'email_reminder' => array(
                            'name' 	=> __( 'Enable email reminder:', 'automatorwp-pro' ),
                            'type' 	=> 'checkbox',
                            'classes' => 'cmb2-switch',
                        ),
                        'time_email_reminder' => array(
                            'name' => __( 'Minutes before event starts:', 'automatorwp-pro' ),
                            'desc' => __( 'An email reminder will be sent minutes before the event starts (15 minutes by default).', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                            ),
                            'default' => '15'
                        ),
                        'popup_reminder' => array(
                            'name' 	=> __( 'Enable popup reminder:', 'automatorwp-pro' ),
                            'type' 	=> 'checkbox',
                            'classes' => 'cmb2-switch',
                        ),
                        'time_popup_reminder' => array(
                            'name' => __( 'Minutes before event start:', 'automatorwp-pro' ),
                            'desc' => __( 'A popup reminder will be displayed minutes before the event starts (15 minutes by default).', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                            ),
                            'default' => '15'
                        ),
                   
                   
                    ),
                    )
                ),
          
        ));

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation){

        // Bail if not selected calendar
        if( empty( $action_options['calendar'] ) ) {
            $this->result = __( 'No calendar selected on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Bail if not data to insert
        if( empty( $action_options['event'] ) ) {
            $this->result = __( 'No data found to insert.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $calendar_id = $action_options['calendar'];        
        $events = $action_options['event'];
        $description = $action_options['description'];
        $start_datetime = date( 'c', $action_options['start_datetime'] ); // ISO 8601 date
        $end_datetime = date('c', $action_options['end_datetime']);
        $email_reminder = $action_options['email_reminder'];
        $popup_reminder = $action_options['popup_reminder'];

        // Attendees
        $attendees = array();
        $event_attendees = array();

        if ( ! empty ( $action_options['attendees'] ) ) {
            $attendees = explode( ',', $action_options['attendees'] );
            
            foreach ( $attendees as $attendee ) {
                $event_attendees[] = array(
                    'email' => trim( $attendee )
                );
            }
        }
        
        $params = automatorwp_google_calendar_get_request_parameters();

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            return;
        }

        // Setup the request parameters
        
        $body_params = array(
            'summary' => $events,
            'description' => $description,
            'start' => array(
                'dateTime' => $start_datetime,
            ),
            'end' => array(
                'dateTime' => $end_datetime,
            ),
            'attendees' => $event_attendees,
        );

        $reminders = array();

        // Send email reminder YES or NO
        $send_email_reminder = ( $email_reminder === 'on' ? 'true' : 'false' );
        $time_email_reminder = ( $send_email_reminder === 'true' ? absint( $action_options['time_email_reminder'] ) : 0 );

        if ( $send_email_reminder === 'true' ){
            $reminders[] = array('method' => 'email', 'minutes' => $time_email_reminder);
        }

        // Send popup reminder YES or NO
        $send_popup_reminder = ( $popup_reminder === 'on' ? 'true' : 'false' );
        $time_popup_reminder = ( $send_popup_reminder === 'true' ? absint( $action_options['time_popup_reminder'] ) : 0 );

        if ( $send_popup_reminder === 'true' ){
            $reminders[] = array('method' => 'popup', 'minutes' => $time_popup_reminder);
        }

        // Add the selected reminders to the query
        if ( $send_email_reminder === 'true' || $send_popup_reminder === 'true' ){
            $array_reminders = array(
                    'useDefault' => FALSE,
                    'overrides' => $reminders,
                 );
            
            $body_params['reminders'] = $array_reminders;
            
        }

        $params['body'] = json_encode( $body_params );

        // Query to add record
        
        $url = 'https://www.googleapis.com/calendar/v3/calendars/'. $calendar_id .'/events?sendNotifications=true';        
        $request = wp_remote_post( $url, $params );
        $response = json_decode( wp_remote_retrieve_body( $request ), true );

        if ( isset( $response['error'] ) ) {
        
            if ( $response['error']['code'] === 400 && $response['error']['message'] === 'The specified time range is empty.') {
                $this->result = __( 'The specified datetime range is not correct', 'automatorwp-pro' ); 
                return false;   
        
            }    
        }
        
        $this->result = __( 'Event added to the calendar successfully!', 'automatorwp-pro' );
        
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }
    
}

new AutomatorWP_Google_Calendar_Add_Event();