<?php
/**
 * Functions
 *
 * @package     AutomatorWP\Integrations\HubSpot\Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @return array|false
 */
function automatorwp_hubspot_get_request_parameters() {

    $access_token = get_option( 'automatorwp_hubspot_access_token' );

    if( $access_token === false ) {
        return false;
    }

    return array(
        'user-agent'  => 'AutomatorWP; ' . home_url(),
        'timeout'     => 120,
        'httpversion' => '1.1',
        'headers'     => array(
            'Authorization' => 'Bearer ' . $access_token,
            'Content-Type'  => 'application/json',
            'Accept'  		=> 'application/json'
        )
    );
}

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @return string|false|WP_Error
 */
function automatorwp_hubspot_refresh_token() {

    $refresh_token = get_option( 'automatorwp_hubspot_refresh_token' );

    if( $refresh_token === false ) {
        return false;
    }

    $params = array(
        'body'	=> array(
            'grant_type'	=> 'refresh_token',
            'client_id'		=> AUTOMATORWP_HUBSPOT_CLIENT_ID,
            'client_secret' => AUTOMATORWP_HUBSPOT_CLIENT_SECRET,
            'redirect_uri'	=> str_replace( 'http://', 'https://', get_admin_url() ) . 'admin.php?page=automatorwp_settings&tab=opt-tab-hubspot',
            'refresh_token' => $refresh_token
        )
    );

    $response = wp_remote_post( 'https://api.hubapi.com/oauth/v1/token', $params );

    if( is_wp_error( $response ) ) {
        return $response;
    }

    $body = json_decode( wp_remote_retrieve_body( $response ) );

    // Update the access token
    update_option( 'automatorwp_hubspot_access_token', $body->access_token );

    return $body->access_token;

}

/**
 * Filters the HTTP API response immediately before the response is returned.
 *
 * @since 1.0.0
 *
 * @param array  $response    HTTP response.
 * @param array  $parsed_args HTTP request arguments.
 * @param string $url         The request URL.
 *
 * @return array
 */
function automatorwp_hubspot_maybe_refresh_token( $response, $args, $url ) {

    // Ensure to only run this check to on HubSpot request
    if( strpos( $url, 'api.hubapi' ) !== false ) {

        $code = wp_remote_retrieve_response_code( $response );

        if( $code === 401 ) {

            $body = json_decode( wp_remote_retrieve_body( $response ) );

            // Check if api returns the expired authentication message
            if( $body->category === 'EXPIRED_AUTHENTICATION' ) {

                $access_token = automatorwp_hubspot_refresh_token();

                // Send again the request if token gets refreshed successfully
                if( $access_token ) {

                    $args['headers']['Authorization'] = 'Bearer ' . $access_token;

                    $response = wp_remote_request( $url, $args );

                }

            }

        }

    }

    return $response;

}
add_filter( 'http_response', 'automatorwp_hubspot_maybe_refresh_token', 10, 3 );

/**
 * Get all lists
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_hubspot_get_lists() {

    $lists = array();

    $transient = get_transient( 'automatorwp_hubspot_lists' );

    if ( $transient !== false ) {
        return $transient;
    }

    $params = automatorwp_hubspot_get_request_parameters();

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return array();
    }

    $url  = 'https://api.hubapi.com/contacts/v1/lists?count=999';
    $response = wp_remote_get( $url, $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true );

    if( isset( $response['lists'] ) && is_array( $response['lists'] ) ) {
        foreach( $response['lists'] as $list ) {

            if( $list['dynamic'] ) {
                continue;
            }

            $lists[] = array(
                'id' => $list['listId'],
                'name' => $list['name'],
            );
        }
    }

    if( count( $lists ) ) {
        // Set a transient for 10 mins with the lists
        set_transient( 'automatorwp_hubspot_lists', $lists, 10 * 60 );
    }

    return $lists;

}

/**
 * Options callback for select2 fields assigned to lists
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_hubspot_options_cb_list( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any list', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $list_id ) {

            // Skip option none
            if( $list_id === $none_value ) {
                continue;
            }

            $options[$list_id] = automatorwp_hubspot_get_list_title( $list_id );
        }
    }

    return $options;

}

/**
 * Get the list title
 *
 * @since 1.0.0
 *
 * @param int $list_id
 *
 * @return string|null
 */
function automatorwp_hubspot_get_list_title( $list_id ) {

    // Empty title if no ID provided
    if( absint( $list_id ) === 0 ) {
        return '';
    }

    $lists = automatorwp_hubspot_get_lists();
    $list_name = '';

    foreach( $lists as $list ) {

        if( absint( $list['id'] ) === absint( $list_id ) ) {
            $list_name = $list['name'];
            break;
        }

    }

    return $list_name;

}