<?php
/**
 * User Course
 *
 * @package     AutomatorWP\Integrations\LearnPress\Actions\User_Course
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnPress_User_Course extends AutomatorWP_Integration_Action {

    public $integration = 'learnpress';
    public $action = 'learnpress_user_course';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Enroll or unroll user from a course', 'automatorwp-pro' ),
            'select_option'     => __( 'Enroll or unroll user from <strong>a course</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'edit_label'        => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Post title. */
            'log_label'         => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{post}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'enroll'     => __( 'Enroll', 'automatorwp-pro' ),
                                'unroll'    => __( 'Unroll', 'automatorwp-pro' ),
                            ),
                            'default' => 'enroll'
                        ),
                    )
                ),
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Course:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'all courses', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Course ID', 'automatorwp-pro' ),
                    'post_type'         => LP_COURSE_CPT,
                ) ),
            ),
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'unroll' ) {
            return sprintf( __( '%1$s user from %2$s', 'automatorwp-pro' ), '{operation}', '{post}' );
        }

        return $label;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $operation = $action_options['operation'];
        $course_id = $action_options['post'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'enroll';
        }

        $user = learn_press_get_user( $user_id );

        // Bail if can't find the user
        if( ! $user ) {
            return;
        }

        $courses = array();

        // Check specific course
        if( $course_id !== 'any' ) {

            $course = get_post( $course_id );

            // Bail if course doesn't exists
            if( ! $course ) {
                return;
            }

            $courses = array( $course_id );

        }

        switch ( $operation ) {
            case 'enroll':
                // If enrolling to all courses, get all courses
                if( $course_id === 'any' ) {

                    $query = new WP_Query( array(
                        'post_type'		=> LP_COURSE_CPT,
                        'post_status'	=> 'publish',
                        'fields'        => 'ids',
                        'nopaging'      => true,
                    ) );

                    $courses = $query->get_posts();
                }

                // Enroll user in courses
                foreach( $courses as $course_id ) {

                    // Skip if user is already on this course
                    if( $user->has_enrolled_course( $course_id ) ) {
                        continue;
                    }

                    $course = learn_press_get_course( $course_id );

                    // Skip if course not exists
                    if( ! $course ) {
                        continue;
                    }

                    // Skip if course not exists
                    if( ! $course->exists() ) {
                        continue;
                    }

                    // Create a new order
                    $order = new LP_Order();
                    $order->set_customer_note( __( 'Order created by AutomatorWP', 'automatorwp-pro' ) );
                    $order->set_status( LP_ORDER_PENDING );
                    $order->set_user_id( $user_id );
                    $order->set_user_ip_address( learn_press_get_ip() );
                    $order->set_user_agent( learn_press_get_user_agent() );
                    $order->set_created_via( 'AutomatorWP' );
                    $order->set_subtotal( 0 );
                    $order->set_total( 0 );

                    // Save the order
                    $order_id = $order->save();

                    // Add the course as order item
                    $order_item = array(
                        'order_item_name'  => $course->get_title(),
                        'item_id'          => $course_id,
                        'quantity'         => 1,
                        'subtotal'         => 0,
                        'total'            => 0,
                    );

                    // Save the order item
                    $item_id = $order->add_item( $order_item, 1 );

                    // Force the order status update
                    $order->update_status( 'completed' );

                }
                break;
            case 'unroll':
                // If unrolling from all courses, get all user courses
                // TODO: LearnPress doesn't offers a method to get user courses
                if( $course_id === 'any' ) {

                    $query = new WP_Query( array(
                        'post_type'		=> LP_COURSE_CPT,
                        'post_status'	=> 'publish',
                        'fields'        => 'ids',
                        'nopaging'      => true,
                    ) );

                    $courses = $query->get_posts();

                }

                // Unroll user from courses
                foreach( $courses as $course_id ) {
                    learn_press_delete_user_data( $user_id, $course_id );
                }
                break;
        }

    }

}

new AutomatorWP_LearnPress_User_Course();