<?php
/**
 * Add Contact
 *
 * @package     AutomatorWP\Integrations\Mautic\Actions\Add_Contact
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Mautic_Add_Contact extends AutomatorWP_Integration_Action {

    public $integration = 'mautic';
    public $action = 'mautic_add_contact';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add contact to Mautic', 'automatorwp-pro' ),
            'select_option'     => __( 'Add <strong>contact</strong> to Mautic', 'automatorwp-pro' ),
            /* translators: %1$s: Contact. */
            'edit_label'        => sprintf( __( 'Add %1$s to Mautic', 'automatorwp-pro' ), '{mautic_fields}' ),
            /* translators: %1$s: Contact. */
            'log_label'         => sprintf( __( 'Add %1$s to Mautic', 'automatorwp-pro' ), '{mautic_fields}' ),
            'options'           => array(
                'mautic_fields' => array(
                    'from' => 'email',
                    'default' => __( 'contact', 'automatorwp-google-sheets' ),
                    'fields' => array(
                        'email' => array(
                            'name' => __( 'Email:', 'automatorwp-pro' ),
                            'desc' => __( 'The contact email address.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required'  => true,
                            'default' => ''
                        ),
                        'mautic_fields' => array(
                            'name' => __( 'Fields:', 'automatorwp-google-sheets' ),
                            'desc' => __( 'Contact data fields.', 'automatorwp-google-sheets' ),
                            'type' => 'group',
                            'classes' => 'automatorwp-fields-table automatorwp-fields-table-col-1',
                            'options'     => array(
                                'add_button'        => __( 'Add field', 'automatorwp-google-sheets' ),
                                'remove_button'     => '<span class="dashicons dashicons-no-alt"></span>',
                                ),
                            'fields' => array(
                                'field' => automatorwp_utilities_ajax_selector_field( array(
                                    'field'             => 'Field',
                                    'name'              => __( 'Field:', 'automatorwp-pro' ),
                                    'option_default'    => __( 'Select a field', 'automatorwp-pro' ),
                                    'action_cb'         => 'automatorwp_mautic_get_contact_fields',
                                    'options_cb'        => 'automatorwp_mautic_options_cb_contact_field',
                                    'default'           => ''
                                ) ),
                                'value' => array(
                                    'name' => __( 'Value:', 'automatorwp-google-sheets' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                            ),
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Bail if Mautic not configured
        $params = automatorwp_mautic_get_request_parameters( );

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            $this->result = __( 'Could not add contact, Mautic authentication failed', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $contact_email = $action_options['email'];
        $mautic_fields = $action_options['mautic_fields'];

        $contact_data = array(
            'email'     => $contact_email,
        );

        // Add fields to contact data        
        foreach ( $mautic_fields as $field ){
            $alias = automatorwp_mautic_get_contact_field_alias( $field['field'] );
            $contact_data[$alias] = $field['value'];
        }
        
        $code = automatorwp_mautic_add_contact( $contact_data );
        
        if ( $code !== 201 ) {
            $this->result = sprintf( __( 'Error %d. %s not added to Mautic', 'automatorwp-pro' ), $code, $contact_email );
        } else {
            $this->result = sprintf( __( '%s added to Mautic', 'automatorwp-pro' ), $contact_email );
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        $params = automatorwp_mautic_get_request_parameters( );

        // Warn user if the authorization has not been setup from settings
        if( $params === false ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Mautic settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-mautic'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/mautic/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Mautic_Add_Contact();