<?php
/**
 * Remove Access Level
 *
 * @package     AutomatorWP\Integrations\Restrict_Content\Actions\Remove_User_Level
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Restrict_Content_Remove_User_Level extends AutomatorWP_Integration_Action {

    public $integration = 'restrict_content';
    public $action = 'restrict_content_remove_access_level';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Remove user from level', 'automatorwp-pro' ),
            'select_option'     => __( 'Remove user from <strong>access level</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Level. */
            'edit_label'        => sprintf( __( 'Remove user from %1$s', 'automatorwp-pro' ), '{access_level}' ),
            /* translators: %1$s: Level. */
            'log_label'         => sprintf( __( 'Remove user from %1$s', 'automatorwp-pro' ), '{access_level}' ),
            'options'           => array(
                'access_level' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'access_level',
                    'option_default'    => __( 'access_level', 'automatorwp-pro' ),
                    'name'              => __( 'Access Level:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Access Level ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_restrict_content_get_access_levels',
                    'options_cb'        => 'automatorwp_restrict_content_options_cb_access_level',
                    'placeholder'       => __( 'Select an access level', 'automatorwp-pro' ),
                    'default'           => '',
                ) )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Shorthand
        $access_level_id = absint( $action_options['access_level'] );

        // Bail if level not provided
        if( empty( $access_level_id ) ) {
            return;
        }

        // Remove the user from level
        automatorwp_restrict_content_remove_user_from_access_level( $user_id, $access_level_id );

        $this->result = __( 'User removed from level successfully.', 'automatorwp-pro' );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }    

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }    

}

new AutomatorWP_Restrict_Content_Remove_User_Level();