<?php
/**
 * RSVP User
 *
 * @package     AutomatorWP\Integrations\The_Events_Calendar\Actions\RSVP_User
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_The_Events_Calendar_RSVP_User extends AutomatorWP_Integration_Action {

    public $integration = 'the_events_calendar';
    public $action = 'the_events_calendar_rsvp_user';

    /**
     * The action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'RSVP user to an event', 'automatorwp-pro' ),
            'select_option'     => __( 'RSVP user to <strong>an event</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. */
            'edit_label'        => sprintf( __( 'RSVP user to %1$s', 'automatorwp-pro' ), '{post}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'RSVP user to %1$s', 'automatorwp-pro' ), '{post}' ),
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name'              => __( 'Events:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'all events', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Event ID', 'automatorwp-pro' ),
                    'post_type'         => Tribe__Events__Main::POSTTYPE,
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Shorthand
        $event_id = $action_options['post'];

        $events = array();

        // Check specific event
        if( $event_id !== 'any' ) {

            $event = get_post( $event_id );

            // Bail if event doesn't exists
            if( ! $event ) {
                $this->result = sprintf( __( 'Could not find the event by the ID %d.', 'automatorwp-pro' ), $event_id );
                return;
            }

            $events = array( $event_id );

        } else if( $event_id === 'any' ) {

            // Get all events
            $query = new WP_Query( array(
                'post_type'		=> Tribe__Events__Main::POSTTYPE,
                'post_status'	=> 'publish',
                'fields'        => 'ids',
                'nopaging'      => true,
            ) );

            $events = $query->get_posts();

        }

        // Bail if no events found to RSVP
        if( empty( $events ) ) {
            $this->result = __( 'Not events found.', 'automatorwp-pro' );
            return;
        }

        // Setup the tickets handler and user details (to be used on attendee details)
        $ticket_handler = new Tribe__Tickets__Tickets_Handler();
        $user = get_userdata( $user_id );

        // RSVP user to events
        foreach( $events as $event_id ) {

            $rsvp_tickets = $ticket_handler->get_event_rsvp_tickets( $event_id );

            // Bail if event hasn't tickets available
            if ( empty( $rsvp_tickets ) ) {
                $this->result .= sprintf( __( 'Event #%d excluded because do not have tickets available.', 'automatorwp-pro' ), $event_id ) . '<br>';
                continue;
            }

            $product_id = false;

            // Loop all event tickets
            foreach ( $rsvp_tickets as $rsvp_ticket ) {

                if ( $rsvp_ticket->capacity < 0 ) {

                    // Use this ticket if not has capacity limit
                    $product_id = $rsvp_ticket->ID;

                } elseif ( $rsvp_ticket->capacity > 0 && $rsvp_ticket->capacity > $rsvp_ticket->qty_sold ) {

                    // Use this ticket if has capacity but the quantity sold doesn't exceeds the ticket capacity
                    $product_id = $rsvp_ticket->ID;

                }

                // Product ID found so exit loop
                if ( $product_id ) {
                    break;
                }
            }

            // Setup the attendee details
            $attendee_details = array(
                'full_name'    => $user->display_name,
                'email'        => $user->user_email,
                'order_status' => 'yes',
                'optout'       => false,
                'order_id'     => '-1',
            );

            // Generate a ticket to the event for this user
            $rsvp = new Tribe__Tickets__RSVP();
            $rsvp->generate_tickets_for( $product_id, 1, $attendee_details );

            $this->result .= sprintf( __( 'Ticket generated for event #%d successfully.', 'automatorwp-pro' ), $event_id ) . '<br>';

        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_The_Events_Calendar_RSVP_User();