<?php
/**
 * User Product
 *
 * @package     AutomatorWP\Integrations\Thrive_Apprentice\Actions\User_Product
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Thrive_Apprentice_User_Product extends AutomatorWP_Integration_Action {

    public $integration = 'thrive_apprentice';
    public $action = 'thrive_apprentice_user_product';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Enroll or unroll user from a product', 'automatorwp-pro' ),
            'select_option'     => __( 'Enroll or unroll user from <strong>a product</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Product title. */
            'edit_label'        => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{term}' ),
            /* translators: %1$s: Operation (add or remove). %2$s: Product title. */
            'log_label'         => sprintf( __( '%1$s user to %2$s', 'automatorwp-pro' ), '{operation}', '{term}' ),
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'enroll'     => __( 'Enroll', 'automatorwp-pro' ),
                                'unroll'    => __( 'Unroll', 'automatorwp-pro' ),
                            ),
                            'default' => 'enroll'
                        ),
                    )
                ),
                'term' => automatorwp_utilities_term_option( array(
                    'name'                  => __( 'Product:', 'automatorwp-pro' ),
                    'option_none_label'     => __( 'all products', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Product ID', 'automatorwp-pro' ),
                    'taxonomy'              => 'tva_product',
                ) ),
            ),
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Dynamic edit and log labels
        add_filter( 'automatorwp_parse_automation_item_edit_label', array( $this, 'dynamic_label' ), 10, 5 );
        add_filter( 'automatorwp_parse_automation_item_log_label', array( $this, 'dynamic_label' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Custom edit/log label
     *
     * @since 1.0.0
     *
     * @param string    $label      The edit label
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The item type (trigger|action)
     * @param string    $context    The context this function is executed
     * @param array     $type_args  The type parameters
     *
     * @return string
     */
    public function dynamic_label( $label, $object, $item_type, $context, $type_args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $label;
        }

        // Get the operation value
        ct_setup_table( "automatorwp_{$item_type}s" );
        $operation = ct_get_object_meta( $object->id, 'operation', true );
        ct_reset_setup_table();

        // Update the edit label
        if( $operation === 'unroll' ) {
            return sprintf( __( '%1$s user from %2$s', 'automatorwp-pro' ), '{operation}', '{term}' );
        }

        return $label;

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        global $wpdb;

        // Shorthand
        $operation = $action_options['operation'];
        $product_id = $action_options['term'];

        // Ensure operation default value
        if( empty( $operation ) ) {
            $operation = 'enroll';
        }

        $products = array();

        // Check specific product
        if( $product_id !== 'any' ) {

            $product = get_term( $product_id, 'tva_product' );

            // Bail if course doesn't exists
            if( ! $product ) {
                return;
            }

            $products = array( $product_id );

        } else {

            $terms = get_terms( array( 
                'taxonomy' => 'tva_product',
                'hide_empty' => false,
            ) );

            foreach ( $terms as $term ) {
                $products[] = $term->term_id;
            }

        }
 
        switch ( $operation ) {
            case 'enroll':
                // Enroll user in products
                foreach( $products as $product_id ) {
                    TVA_Customer::enrol_user_to_product( $user_id, $product_id );
                }
                break;
            case 'unroll':
                // Unroll user in products
                foreach( $products as $product_id ) {
                    TVA_Customer::remove_user_from_product( $user_id, $product_id );
                }
                break;
        }

    }

}

new AutomatorWP_Thrive_Apprentice_User_Product();