<?php
/**
 * Get Like Activity
 *
 * @package     AutomatorWP\Integrations\WP_Ulike\Triggers\Get_Like_Activity
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WP_Ulike_Get_Like_Activity extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_ulike';
    public $trigger = 'wp_ulike_get_like_activity';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets a like or unlike on an activity', 'automatorwp-pro' ),
            'select_option'     => __( 'User gets a like or unlike on <strong>an activity</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Operation (likes or unlikes). Number of times. */
            'edit_label'        => sprintf( __( 'User gets a %1$s on an activity %2$s time(s)', 'automatorwp-pro' ), '{operation}', '{times}' ),
            /* translators: %1$s: Operation (likes or unlikes). */
            'log_label'         => sprintf( __( 'User gets a %1$s on an activity', 'automatorwp-pro' ), '{operation}' ),
            'action'            => 'wp_ulike_after_process',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 4,
            'options'           => array(
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'like'      => __( 'like', 'automatorwp-pro' ),
                                'unlike'    => __( 'unlike', 'automatorwp-pro' ),
                            ),
                            'default' => 'like'
                        ),
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $id         Liked element ID
     * @param string    $key        Like key
     * @param int       $user_id    User ID
     * @param string    $status     like|unlike
     */
    public function listener( $id, $key, $user_id, $status ) {

        // Bail if not is a post like
        if( $key !== '_activityliked' ) {
            return;
        }

        // Get the author ID since is who receives the like or unlike
        $activity = bp_activity_get_specific( array( 'activity_ids' => $id, 'display_comments'  => true ) );

        $author_id = 0;

        if( isset( $activity['activities'][0] ) ) {
            $author_id = $activity['activities'][0]->user_id;
        }

        if( absint( $author_id ) === 0 ) {
            return;
        }

        // Trigger the get a like or unlike
        automatorwp_trigger_event( array(
            'trigger'   => $this->trigger,
            'user_id'   => $author_id,
            'status'    => $status,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Shorthand
        $operation = $trigger_options['operation'];

        if( empty( $operation ) ) {
            $operation = 'like';
        }

        // Don't deserve if status is not received
        if( ! isset( $event['status'] ) ) {
            return false;
        }

        switch( $operation ) {
            case 'like':
                // Don't deserve if user hasn't liked
                if( $event['status'] !== 'like' ) {
                    return false;
                }
                break;
            case 'unlike':
                // Don't deserve if user hasn't unliked
                if( $event['status'] === 'like' ) {
                    return false;
                }
                break;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WP_Ulike_Get_Like_Activity();