<?php
/**
 * Complete Assignment Mark
 *
 * @package     AutomatorWP\Integrations\WPLMS\Triggers\Complete_Assignment_Mark
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_WPLMS_Complete_Assignment_Mark extends AutomatorWP_Integration_Trigger {

    public $integration = 'wplms';
    public $trigger = 'wplms_complete_assignment_mark';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes a assignment with a mark greater than, less than or equal to a specific mark', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes a assignment with a mark <strong>greater than, less than or equal</strong> to a specific mark', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Percent. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes %1$s with a mark %2$s %3$s %4$s time(s)', 'automatorwp-pro' ), '{post}', '{condition}', '{mark}', '{times}' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Percent. */
            'log_label'         => sprintf( __( 'User completes %1$s with a mark %2$s %3$s', 'automatorwp-pro' ), '{post}', '{condition}', '{mark}' ),
            'action'            => 'wplms_evaluate_assignment',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Assignment:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any assignment', 'automatorwp-pro' ),
                    'post_type' => 'assignment'
                ) ),
                'condition' => automatorwp_utilities_condition_option(),
                'mark' => array(
                    'from' => 'mark',
                    'fields' => array(
                        'mark' => array(
                            'name' => __( 'Mark:', 'automatorwp-pro' ),
                            'desc' => __( 'The mark required.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param integer $assignment_id
     * @param string $marks             Can be a single number of a range separated by "-"
     * @param integer $user_id
     */
    public function listener( $assignment_id, $marks, $user_id ) {

        if( ! $user_id ) {
            $user_id = get_current_user_id();
        }

        // Get the mark
        if( is_numeric( $marks ) ) {
            // Numeric score (X)
            $mark = absint( $marks );
        } else if( strpos( $marks, '-' ) !== false ) {
            // Score range (X-Y)
            $range = explode( '-', $marks );

            $mark = $range[1];
        }

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'post_id'       => $assignment_id,
            'mark'          => $mark,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['mark'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        $mark = (float) $event['mark'];
        $required_mark =  (float) $trigger_options['mark'];

        // Don't deserve if order total doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $mark, $required_mark, $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_WPLMS_Complete_Assignment_Mark();