import {commonAdminQueries} from '@common/admin/common-admin-queries';
import {GlobalLoadingProgress} from '@common/core/global-loading-progress';
import {DataTableHeader} from '@common/datatable/data-table-header';
import {DataTablePaginationFooter} from '@common/datatable/data-table-pagination-footer';
import {useDatatableSearchParams} from '@common/datatable/filters/utils/use-datatable-search-params';
import {validateDatatableSearch} from '@common/datatable/filters/utils/validate-datatable-search';
import {DatatableFilters} from '@common/datatable/page/datatable-filters';
import {
  DatatablePageHeaderBar,
  DatatablePageScrollContainer,
  DatatablePageWithHeaderBody,
  DatatablePageWithHeaderLayout,
} from '@common/datatable/page/datatable-page-with-header-layout';
import {useDatatableQuery} from '@common/datatable/requests/use-datatable-query';
import {Table} from '@common/ui/tables/table';
import {IconButton} from '@ui/buttons/icon-button';
import {Chip} from '@ui/forms/input-field/chip-field/chip';
import {FormattedDate} from '@ui/i18n/formatted-date';
import {Trans} from '@ui/i18n/trans';
import {CloseIcon} from '@ui/icons/material/Close';
import {EditIcon} from '@ui/icons/material/Edit';
import {PauseIcon} from '@ui/icons/material/Pause';
import {PlayArrowIcon} from '@ui/icons/material/PlayArrow';
import {ConfirmationDialog} from '@ui/overlays/dialog/confirmation-dialog';
import {DialogTrigger} from '@ui/overlays/dialog/dialog-trigger';
import {Tooltip} from '@ui/tooltip/tooltip';
import {Fragment} from 'react';
import {useCancelSubscription} from '../../billing/billing-page/requests/use-cancel-subscription';
import {useResumeSubscription} from '../../billing/billing-page/requests/use-resume-subscription';
import {Subscription} from '../../billing/subscription';
import {ColumnConfig} from '../../datatable/column-config';
import {NameWithAvatar} from '../../datatable/column-templates/name-with-avatar';
import {DataTableAddItemButton} from '../../datatable/data-table-add-item-button';
import {DataTableEmptyStateMessage} from '../../datatable/page/data-table-emty-state-message';
import {queryClient} from '../../http/query-client';
import {CreateSubscriptionDialog} from './create-subscription-dialog';
import {SubscriptionIndexPageFilters} from './subscription-index-page-filters';
import subscriptionsSvg from './subscriptions.svg';
import {UpdateSubscriptionDialog} from './update-subscription-dialog';

const endpoint = 'billing/subscriptions';

const columnConfig: ColumnConfig<Subscription>[] = [
  {
    key: 'user_id',
    allowsSorting: true,
    width: 'flex-3 min-w-200',
    visibleInMode: 'all',
    header: () => <Trans message="Customer" />,
    body: subscription =>
      subscription.user && (
        <NameWithAvatar
          image={subscription.user.image}
          label={subscription.user.name}
          description={subscription.user.email}
        />
      ),
  },
  {
    key: 'status',
    width: 'w-100 flex-shrink-0',
    header: () => <Trans message="Status" />,
    body: subscription => (
      <Chip
        size="xs"
        color={subscription.valid ? 'positive' : 'danger'}
        radius="rounded"
        className="w-max"
      >
        {subscription.gateway_status}
      </Chip>
    ),
  },
  {
    key: 'product_id',
    allowsSorting: true,
    header: () => <Trans message="Plan" />,
    body: subscription => subscription.product?.name,
  },
  {
    key: 'gateway_name',
    allowsSorting: true,
    header: () => <Trans message="Gateway" />,
    body: subscription => (
      <span className="capitalize">{subscription.gateway_name}</span>
    ),
  },
  {
    key: 'renews_at',
    allowsSorting: true,
    header: () => <Trans message="Renews at" />,
    body: subscription => <FormattedDate date={subscription.renews_at} />,
  },
  {
    key: 'ends_at',
    allowsSorting: true,
    header: () => <Trans message="Ends at" />,
    body: subscription => <FormattedDate date={subscription.ends_at} />,
  },
  {
    key: 'created_at',
    allowsSorting: true,
    header: () => <Trans message="Created at" />,
    body: subscription => <FormattedDate date={subscription.created_at} />,
  },
  {
    key: 'actions',
    header: () => <Trans message="Actions" />,
    hideHeader: true,
    align: 'end',
    visibleInMode: 'all',
    width: 'w-[168px] flex-shrink-0',
    body: subscription => {
      return <SubscriptionActions subscription={subscription} />;
    },
  },
];

export function Component() {
  const {
    searchParams,
    sortDescriptor,
    mergeIntoSearchParams,
    setSearchQuery,
    isFiltering,
  } = useDatatableSearchParams(validateDatatableSearch);

  const query = useDatatableQuery(
    commonAdminQueries.subscriptions.index(searchParams),
  );

  const actions = (
    <DialogTrigger type="modal">
      <DataTableAddItemButton>
        <Trans message="Add new subscription" />
      </DataTableAddItemButton>
      <CreateSubscriptionDialog />
    </DialogTrigger>
  );

  return (
    <DatatablePageWithHeaderLayout>
      <GlobalLoadingProgress query={query} />
      <DatatablePageHeaderBar
        title={<Trans message="Subscriptions" />}
        showSidebarToggleButton
      />
      <DatatablePageWithHeaderBody>
        <DataTableHeader
          searchValue={searchParams.query}
          onSearchChange={setSearchQuery}
          actions={actions}
          filters={SubscriptionIndexPageFilters}
        />
        <DatatableFilters filters={SubscriptionIndexPageFilters} />
        <DatatablePageScrollContainer>
          <Table
            columns={columnConfig}
            data={query.items}
            sortDescriptor={sortDescriptor}
            onSortChange={mergeIntoSearchParams}
            enableSelection={false}
          />
          {query.isEmpty && (
            <DataTableEmptyStateMessage
              className="mt-50"
              isFiltering={isFiltering}
              image={subscriptionsSvg}
              title={<Trans message="No subscriptions have been created yet" />}
              filteringTitle={<Trans message="No matching subscriptions" />}
            />
          )}
          <DataTablePaginationFooter
            query={query}
            onPageChange={page => mergeIntoSearchParams({page})}
            onPerPageChange={perPage => mergeIntoSearchParams({perPage})}
          />
        </DatatablePageScrollContainer>
      </DatatablePageWithHeaderBody>
    </DatatablePageWithHeaderLayout>
  );
}

interface SubscriptionActionsProps {
  subscription: Subscription;
}
function SubscriptionActions({subscription}: SubscriptionActionsProps) {
  return (
    <Fragment>
      <DialogTrigger type="modal">
        <IconButton size="md" className="text-muted">
          <EditIcon />
        </IconButton>
        <UpdateSubscriptionDialog subscription={subscription} />
      </DialogTrigger>
      {subscription.cancelled && subscription.on_grace_period ? (
        <ResumeSubscriptionButton subscription={subscription} />
      ) : null}
      {subscription.active ? (
        <SuspendSubscriptionButton subscription={subscription} />
      ) : null}
      <CancelSubscriptionButton subscription={subscription} />
    </Fragment>
  );
}

function SuspendSubscriptionButton({subscription}: SubscriptionActionsProps) {
  const cancelSubscription = useCancelSubscription();

  const handleSuspendSubscription = () => {
    cancelSubscription.mutate(
      {subscriptionId: subscription.id},
      {
        onSuccess: () => {
          queryClient.invalidateQueries({
            queryKey: commonAdminQueries.subscriptions.invalidateKey,
          });
        },
      },
    );
  };

  return (
    <DialogTrigger
      type="modal"
      onClose={confirmed => {
        if (confirmed) {
          handleSuspendSubscription();
        }
      }}
    >
      <Tooltip label={<Trans message="Cancel subscription" />}>
        <IconButton
          size="md"
          className="text-muted"
          disabled={cancelSubscription.isPending}
        >
          <PauseIcon />
        </IconButton>
      </Tooltip>
      <ConfirmationDialog
        title={<Trans message="Cancel subscription" />}
        body={
          <div>
            <Trans message="Are you sure you want to cancel this subscription?" />
            <div className="mt-10 text-sm font-semibold">
              <Trans message="This will put user on grace period until their next scheduled renewal date. Subscription can be renewed until that date by user or from admin area." />
            </div>
          </div>
        }
        confirm={<Trans message="Confirm" />}
      />
    </DialogTrigger>
  );
}

function ResumeSubscriptionButton({subscription}: SubscriptionActionsProps) {
  const resumeSubscription = useResumeSubscription();
  const handleResumeSubscription = () => {
    resumeSubscription.mutate(
      {subscriptionId: subscription.id},
      {
        onSuccess: () => {
          queryClient.invalidateQueries({
            queryKey: commonAdminQueries.subscriptions.invalidateKey,
          });
        },
      },
    );
  };

  return (
    <DialogTrigger
      type="modal"
      onClose={confirmed => {
        if (confirmed) {
          handleResumeSubscription();
        }
      }}
    >
      <Tooltip label={<Trans message="Renew subscription" />}>
        <IconButton
          size="md"
          className="text-muted"
          onClick={handleResumeSubscription}
          disabled={resumeSubscription.isPending}
        >
          <PlayArrowIcon />
        </IconButton>
      </Tooltip>
      <ConfirmationDialog
        title={<Trans message="Resume subscription" />}
        body={
          <div>
            <Trans message="Are you sure you want to resume this subscription?" />
            <div className="mt-10 text-sm font-semibold">
              <Trans message="This will put user on their original plan and billing cycle." />
            </div>
          </div>
        }
        confirm={<Trans message="Confirm" />}
      />
    </DialogTrigger>
  );
}

function CancelSubscriptionButton({subscription}: SubscriptionActionsProps) {
  const cancelSubscription = useCancelSubscription();

  const handleDeleteSubscription = () => {
    cancelSubscription.mutate(
      {subscriptionId: subscription.id, delete: true},
      {
        onSuccess: () => {
          queryClient.invalidateQueries({
            queryKey: commonAdminQueries.subscriptions.invalidateKey,
          });
        },
      },
    );
  };

  return (
    <DialogTrigger
      type="modal"
      onClose={confirmed => {
        if (confirmed) {
          handleDeleteSubscription();
        }
      }}
    >
      <Tooltip label={<Trans message="Delete subscription" />}>
        <IconButton
          size="md"
          className="text-muted"
          disabled={cancelSubscription.isPending}
        >
          <CloseIcon />
        </IconButton>
      </Tooltip>
      <ConfirmationDialog
        isDanger
        title={<Trans message="Delete subscription" />}
        body={
          <div>
            <Trans message="Are you sure you want to delete this subscription?" />
            <div className="mt-10 text-sm font-semibold">
              <Trans message="This will permanently delete the subscription and immediately cancel it on billing gateway. Subscription will not be renewable anymore." />
            </div>
          </div>
        }
        confirm={<Trans message="Confirm" />}
      />
    </DialogTrigger>
  );
}
