<?php
namespace BetterLinksPro\API;

use BetterLinks\Traits\ArgumentSchema;
use BetterLinksPro\Helper;

class Clicks
{
    use ArgumentSchema;
    private $namespace;
    public function __construct()
    {
        $this->namespace = BETTERLINKS_PLUGIN_SLUG . '/v1';
        add_action('betterlinks_register_clicks_routes', array($this, 'register_routes'));
    }

    public function register_routes()
    {
        $endpoint = '/clicks/';
        register_rest_route(
            $this->namespace,
            '/clicks/individual/(?P<id>[\d]+)',
            array(
                'args'   => array(
                    'id' => array(
                        'description' => __('Unique identifier for the object.'),
                        'type'        => 'integer',
                    ),
                ),
                array(
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_item' ),
                    'permission_callback' => [$this, 'permissions_check'],
                    'args'                => $this->get_clicks_schema(),
                ),
            )
        );
        register_rest_route(
            $this->namespace,
            '/clicks/splittest/(?P<id>[\d]+)',
            array(
                'args'   => array(
                    'id' => array(
                        'description' => __('Unique identifier for the object.'),
                        'type'        => 'integer',
                    ),
                ),
                array(
                    'methods'             => \WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_split_test_item' ),
                    'permission_callback' => [$this, 'permissions_check'],
                    'args'                => $this->get_clicks_schema(),
                ),
            )
        );

        register_rest_route(
			$this->namespace,
			$endpoint . 'get_medium/',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_medium' ),
					'permission_callback' => array( $this, 'permissions_check' ),
					'args'                => $this->get_clicks_schema(),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			$endpoint . 'get_charts/',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_charts' ),
					'permission_callback' => array( $this, 'permissions_check' ),
					'args'                => $this->get_clicks_schema(),
				),
			)
		);
    }

    public function get_item($request)
    {
        $request = $request->get_params();
        $results = \BetterLinksPro\Helper::get_individual_link_analytics($request);
        return new \WP_REST_Response(
            [
                'success' => true,
                'data' => $results,
            ],
            200
        );
    }
    
    public function get_split_test_item($request)
    {
        $request = $request->get_params();
        $results = \BetterLinksPro\Helper::get_split_test_analytics_data($request);
        return new \WP_REST_Response(
            [
                'success' => true,
                'data' => $results,
            ],
            200
        );
    }

    /**
     * Check if a given request has access to update a setting
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|bool
     */
    public function permissions_check($request)
    {
        return apply_filters('betterlinkspro/api/analytics_items_permissions_check', current_user_can('manage_options'));
    }

    /**
	 * Get Analytics Top Medium Data
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_medium( $request ) {
		$request = $request->get_params();
		$from    = isset( $request['from'] ) ? $request['from'] : date( 'Y-m-d', strtotime( ' - 30 days' ) );
		$to      = isset( $request['to'] ) ? $request['to'] : date( 'Y-m-d' );

		$top_medium = array();
		if ( apply_filters( 'betterlinks/is_extra_data_tracking_compatible', false ) ) {
			$all_referer = Helper::get_all_referer( $from, $to );
			$top_medium  = Helper::get_top_medium( $all_referer );
		}
		return new \WP_REST_Response(
			array(
				'success' => true,
				'data'    => array(
					'medium' => $top_medium,
				),
			)
		);
	}

	/**
	 * Get Analytics Chart Data
	 *
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_charts( $request ) {
		$request = $request->get_params();
		$from    = isset( $request['from'] ) ? $request['from'] : date( 'Y-m-d', strtotime( ' - 30 days' ) );
		$to      = isset( $request['to'] ) ? $request['to'] : date( 'Y-m-d' );

		$top_referer = $device_stats = $top_os = $top_browser = array();
		if ( apply_filters( 'betterlinks/is_extra_data_tracking_compatible', false ) ) {
			$top_referer  = Helper::get_top_referer( $from, $to );
			$device_stats = Helper::get_device_click_stats( $from, $to );
			$top_os       = Helper::get_top_os( $from, $to );
			$top_browser  = Helper::prepare_browser_data( $from, $to );
		}

		return new \WP_REST_Response(
			array(
				'success' => true,
				'data'    => array(
					'referer' => $top_referer,
					'devices' => $device_stats,
					'os'      => $top_os,
					'browser' => $top_browser,
				),
			)
		);
	}
}
