<?php

namespace BetterLinksPro\Frontend;

class AutoLinks {

	private $protected_tags_content_lists = array();
	private $unique_number;
	private $empty_placeholder;
	private $is_show_icon = false;
	private $custom_icon_svg = '';
	private $disable_autolink = false;
	private $keyword_styles = array();
	private $global_alk_settings = array();
	private $enable_custom_styles = false;

	public static function init() {
		$self = new self();
		global $betterlinks_settings;
		$self->disable_autolink = isset( $betterlinks_settings['disable_autolink'] ) ? $betterlinks_settings['disable_autolink'] : false;
		if( !empty( $self->disable_autolink ) ) return;
		
		// Check if custom styling is enabled (default to false to inherit theme styles)
		$self->enable_custom_styles = isset( $betterlinks_settings['enable_autolink_styles'] ) ? $betterlinks_settings['enable_autolink_styles'] : false;
		
		$self->is_show_icon = isset( $betterlinks_settings['is_autolink_icon'] ) ? $betterlinks_settings['is_autolink_icon'] : false;
		
		// Get custom icon if available
		$self->custom_icon_svg = isset( $betterlinks_settings['autolink_custom_icon'] ) ? $betterlinks_settings['autolink_custom_icon'] : '';
		
		// Get global auto-link keyword settings including HTML options
		$alk_settings = isset( $betterlinks_settings['alk'] ) ? $betterlinks_settings['alk'] : array();
		$self->global_alk_settings = array(
			'openNewTab' => isset( $alk_settings['openNewTab'] ) ? $alk_settings['openNewTab'] : false,
			'useNoFollow' => isset( $alk_settings['useNoFollow'] ) ? $alk_settings['useNoFollow'] : false,
			'caseSensitive' => isset( $alk_settings['caseSensitive'] ) ? $alk_settings['caseSensitive'] : false
		);
		
		// Get auto-link keyword styling settings - use 'inherit' as default to inherit theme styles
		$alk_styles = isset( $betterlinks_settings['alk']['styles'] ) ? $betterlinks_settings['alk']['styles'] : array();
		$self->keyword_styles = array(
			'textColor' => isset( $alk_styles['textColor'] ) ? $alk_styles['textColor'] : 'inherit',
			'fontWeight' => isset( $alk_styles['fontWeight'] ) ? $alk_styles['fontWeight'] : 'inherit',
			'fontStyle' => isset( $alk_styles['fontStyle'] ) ? $alk_styles['fontStyle'] : 'inherit',
			'fontSize' => isset( $alk_styles['fontSize'] ) ? $alk_styles['fontSize'] : 'inherit',
			'textDecoration' => isset( $alk_styles['textDecoration'] ) ? $alk_styles['textDecoration'] : 'inherit',
			'backgroundColor' => isset( $alk_styles['backgroundColor'] ) ? $alk_styles['backgroundColor'] : 'transparent',
			'hoverTextColor' => isset( $alk_styles['hoverTextColor'] ) ? $alk_styles['hoverTextColor'] : 'inherit'
		);
		
		// Add CSS for auto-link keywords styling
		add_action( 'wp_head', array( $self, 'autolink_css' ) );
		
		if ( $self->is_show_icon ) {
			add_action( 'wp_head', array( $self, 'autolink_icon_css' ) );
		}
		add_filter( 'the_content', array( $self, 'add_autolinks' ), 100 );
		add_filter( 'get_the_excerpt', array( $self, 'add_autolinks' ), 100 );
	}

	/**
	 * Sanitize custom SVG icon for security and apply dynamic sizing
	 * 
	 * @param string $svg_content The SVG content to sanitize
	 * @param int $font_size Font size in pixels for dynamic icon sizing
	 * @return string Sanitized SVG content
	 */
	public static function sanitize_custom_svg( $svg_content, $font_size = 16 ) {
		if ( empty( $svg_content ) ) {
			return '';
		}

		// Remove dangerous elements and attributes for security
		$dangerous_attributes = array( 'onload', 'onerror', 'onclick', 'onmouseover', 'onfocus', 'onblur', 'onchange' );
		$dangerous_elements = array( 'script', 'iframe', 'object', 'embed', 'foreignObject' );
		
		$sanitized = $svg_content;
		
		// Remove dangerous attributes
		foreach ( $dangerous_attributes as $attr ) {
			$sanitized = preg_replace( '/\s*' . preg_quote( $attr, '/' ) . '\s*=\s*["\'][^"\']*["\']/i', '', $sanitized );
		}
		
		// Remove dangerous elements
		foreach ( $dangerous_elements as $element ) {
			$sanitized = preg_replace( '/<' . preg_quote( $element, '/' ) . '[^>]*>.*?<\/' . preg_quote( $element, '/' ) . '>/is', '', $sanitized );
		}

		// Remove external references but keep internal ones (like #clipPath)
		$sanitized = preg_replace( '/xlink:href\s*=\s*["\'][^"\'#][^"\']*["\']/i', '', $sanitized );
		$sanitized = preg_replace( '/href\s*=\s*["\'][^"\'#][^"\']*["\']/i', '', $sanitized );

		// Ensure the SVG has the btl_autolink_icon_svg class for consistent styling
		if ( strpos( $sanitized, 'class=' ) !== false ) {
			// Add to existing class attribute
			$sanitized = preg_replace( '/class=(["\'])([^"\']*)\1/', 'class=$1$2 btl_autolink_icon_svg$1', $sanitized );
		} else {
			// Add class attribute to SVG tag
			$sanitized = preg_replace( '/<svg/', '<svg class="btl_autolink_icon_svg"', $sanitized );
		}

		// Update width and height attributes to match font size dynamically
		$sanitized = preg_replace( '/width\s*=\s*["\'][^"\']*["\']/i', 'width="' . $font_size . 'px"', $sanitized );
		$sanitized = preg_replace( '/height\s*=\s*["\'][^"\']*["\']/i', 'height="' . $font_size . 'px"', $sanitized );
		
		// If no width/height attributes exist, add them
		if ( !preg_match( '/width\s*=/i', $sanitized ) && !preg_match( '/height\s*=/i', $sanitized ) ) {
			$sanitized = preg_replace( '/<svg/', '<svg width="' . $font_size . 'px" height="' . $font_size . 'px"', $sanitized );
		}

		// Use custom wp_kses with SVG allowed elements
		$allowed_svg = array(
			'svg' => array(
				'class' => array(),
				'width' => array(),
				'height' => array(),
				'viewbox' => array(),
				'viewBox' => array(),
				'fill' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'xmlns' => array(),
			),
			'path' => array(
				'd' => array(),
				'stroke' => array(),
				// 'stroke-width' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'fill' => array(),
				'fill-rule' => array(),
				'fill-opacity' => array(),
				'opacity' => array(),
			),
			'g' => array(
				'fill' => array(),
				'fill-opacity' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'clip-path' => array(),
				'transform' => array(),
				'opacity' => array(),
			),
			'circle' => array(
				'cx' => array(),
				'cy' => array(),
				'r' => array(),
				'fill' => array(),
				'fill-opacity' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'transform' => array(),
				'opacity' => array(),
			),
			'rect' => array(
				'x' => array(),
				'y' => array(),
				'width' => array(),
				'height' => array(),
				'rx' => array(),
				'ry' => array(),
				'fill' => array(),
				'fill-opacity' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'transform' => array(),
				'opacity' => array(),
			),
			'line' => array(
				'x1' => array(),
				'y1' => array(),
				'x2' => array(),
				'y2' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'transform' => array(),
				'opacity' => array(),
			),
			'ellipse' => array(
				'cx' => array(),
				'cy' => array(),
				'rx' => array(),
				'ry' => array(),
				'fill' => array(),
				'fill-opacity' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'transform' => array(),
				'opacity' => array(),
			),
			'polygon' => array(
				'points' => array(),
				'fill' => array(),
				'fill-opacity' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'transform' => array(),
				'opacity' => array(),
			),
			'polyline' => array(
				'points' => array(),
				'fill' => array(),
				'fill-opacity' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'stroke-dasharray' => array(),
				'stroke-dashoffset' => array(),
				'stroke-opacity' => array(),
				'transform' => array(),
				'opacity' => array(),
			),
			'defs' => array(),
			'clipPath' => array(
				'id' => array(),
			),
			'mask' => array(
				'id' => array(),
			),
			'use' => array(
				'href' => array(),
				'xlink:href' => array(),
				'x' => array(),
				'y' => array(),
				'width' => array(),
				'height' => array(),
				'transform' => array(),
			),
			'text' => array(
				'x' => array(),
				'y' => array(),
				'dx' => array(),
				'dy' => array(),
				'fill' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'font-family' => array(),
				'font-size' => array(),
				'font-weight' => array(),
				'text-anchor' => array(),
				'transform' => array(),
			),
			'tspan' => array(
				'x' => array(),
				'y' => array(),
				'dx' => array(),
				'dy' => array(),
				'fill' => array(),
				'stroke' => array(),
				'stroke-linecap' => array(),
				'stroke-linejoin' => array(),
				'font-family' => array(),
				'font-size' => array(),
				'font-weight' => array(),
			),
		);

		return wp_kses( $sanitized, $allowed_svg );
	}

	public function add_autolinks( $content ) {
		if ( ! ( strpos( $content, 'class="btl_autolink_hyperlink" ' ) === false ) ) {
			return $content;
		}
		if ( is_attachment() || is_feed() ) {
			return $content;
		}

		// If $content is null, bail out early to preserve original return value and avoid the deprecation.
		if ( null === $content ) {
			return $content;
		}
		$ID        = get_the_ID();
		$post_type = get_post_type( $ID );
		global $betterlinks;
		$autolink_disable_post_types = isset( $betterlinks['autolink_disable_post_types'] ) && is_array( $betterlinks['autolink_disable_post_types'] )
					? $betterlinks['autolink_disable_post_types']
					: array();
		if ( in_array( $post_type, $autolink_disable_post_types, true ) || get_post_meta( $ID, 'betterlinks_is_disable_auto_keyword', true ) ) {
			return $content;
		}
		$this->unique_number = wp_rand( 0, 99999 );
		// placeholder variables
		$btl_plc_space            = '_spt_' . $this->unique_number . '_s___pt_' . $this->unique_number;
		$btl_plc_slash            = '_slsh_' . $this->unique_number . '_sl___sh_' . $this->unique_number;
		$btl_plc_colon            = '_slcln_' . $this->unique_number . '_slc___ln_' . $this->unique_number;
		$btl_plc_svg              = '_svgln_' . $this->unique_number . '_svg___ln_' . $this->unique_number;
		$this->empty_placeholder  = '_mptbtl_' . $this->unique_number . '_mpt___btl_';
		$current_permalink        = get_the_permalink();
		$post_category            = get_the_category( $ID );
		$post_category            = ( ! empty( $post_category ) ? wp_list_pluck( $post_category, 'slug' ) : array() );
		$post_tags                = get_the_tags( $ID );
		$post_tags                = ( ! empty( $post_tags ) ? wp_list_pluck( $post_tags, 'slug' ) : array() );
		$keywords                 = $this->get_keywords();
		$content                  = $this->prevent_protected_tags_contents_from_getting_autolinked( $content );
		$content                  = $this->prevent_all_tag_attributes_from_getting_autolinked( $content );
		$should_use_json_for_link = defined( 'BETTERLINKS_EXISTS_LINKS_JSON' ) && BETTERLINKS_EXISTS_LINKS_JSON && count( $keywords ) > 0;
		$links_formatted_by_id    = array();
		if ( $should_use_json_for_link ) {
			$links                 = isset( $betterlinks['links'] ) ? $betterlinks['links'] : array();
			$ids                   = array_column( $links, 'ID' );
			$links_formatted_by_id = array_combine( $ids, $links );
		}
		$uncloaked_categories = isset( $betterlinks['uncloaked_categories'] ) && is_array( $betterlinks['uncloaked_categories'] )
									? array_map( 'intval', $betterlinks['uncloaked_categories'] )
									: array();
		$uncloaked_cats_count = count( $uncloaked_categories );
		$is_force_https       = isset( $betterlinks['force_https'] ) && $betterlinks['force_https'];

		// Protect placeholder strings from being matched by keywords
		// Replace them with safe placeholders that won't contain characters that could match keywords
		$placeholder_protection_map = array();
		$placeholder_protection_map[ $btl_plc_space ] = '[BTLPLC_SPACE_' . $this->unique_number . ']';
		$placeholder_protection_map[ $btl_plc_slash ] = '[BTLPLC_SLASH_' . $this->unique_number . ']';
		$placeholder_protection_map[ $btl_plc_colon ] = '[BTLPLC_COLON_' . $this->unique_number . ']';
		$placeholder_protection_map[ $btl_plc_svg ] = '[BTLPLC_SVG_' . $this->unique_number . ']';
		$placeholder_protection_map[ $this->empty_placeholder ] = '[BTLPLC_EMPTY_' . $this->unique_number . ']';

		// Protect placeholders before keyword matching
		$content = str_replace( array_keys( $placeholder_protection_map ), array_values( $placeholder_protection_map ), $content );

		foreach ( $keywords as $item ) {
			if (
				// check keyword and link id not empty
				( empty( $item['keywords'] ) && empty( $item['link_id'] ) )
				// check post type
				|| ( ! empty( $item['post_type'] ) && ! in_array( $post_type, $item['post_type'], true ) )
				// check category
				|| ( ! empty( $item['category'] ) && count( array_intersect( $post_category, $item['category'] ) ) === 0 )
				// check tags
				|| ( ! empty( $item['tags'] ) && count( array_intersect( $post_tags, $item['tags'] ) ) === 0 )
			) {
				continue;
			}
			if ( $should_use_json_for_link ) {
				$link_id = $item['link_id'];
				$link    = isset( $links_formatted_by_id[ $link_id ] ) ? $links_formatted_by_id[ $link_id ] : array();
			} else {
				$link = $uncloaked_cats_count > 0
							? current( \BetterLinks\Helper::get_link_data_with_cat_id_by_link_id( $item['link_id'] ) )
							: current( \BetterLinks\Helper::get_link_by_ID( $item['link_id'] ) );
			}
			if (
				// check if shortlink exist
				! isset( $link['short_url'] ) ||
				// check if target_url exist
				! isset( $link['target_url'] ) ||
				// check if in the same page as the target url
				$this->make_url_string_comparable( $link['target_url'] ) === $this->make_url_string_comparable( $current_permalink )
			) {
				continue;
			}
			$tags         = $this->fix_for_apostophie( $item['keywords'] );
			$autolink_url = '';
			if ( ( isset( $link['uncloaked'] ) && $link['uncloaked'] ) || ( isset( $link['cat_id'] ) && in_array( intval( $link['cat_id'] ), $uncloaked_categories, true ) ) ) {
				if ( wp_parse_url( $link['target_url'], PHP_URL_SCHEME ) === null ) {
					$autolink_url = $is_force_https
					? 'https://' . $link['target_url']
					: 'http://' . $link['target_url'];
				} else {
					$autolink_url = $is_force_https
					? preg_replace( '/^https?:\/\//i', 'https://', $link['target_url'] )
					: $link['target_url'];
				}
			} else {
				$autolink_url = \BetterLinks\Helper::generate_short_url( $link['short_url'] );
			}
			$autolink_url = str_replace( array( '/', ':' ), array( $btl_plc_slash, $btl_plc_colon ), $autolink_url );
			$search_mode  = 'iu';

			// Use individual item settings if available, otherwise fallback to global settings
			$case_sensitive = isset( $item['case_sensitive'] ) ? $item['case_sensitive'] : $this->global_alk_settings['caseSensitive'];
			if ( true === (bool) $case_sensitive ) {
				$search_mode = 'u';
			}
			$attribute      = $this->get_link_attributes( $item );
			$keyword_before = ( ! empty( $item['keyword_before'] ) ? $this->fix_for_apostophie( $item['keyword_before'] ) : '' );
			$keyword_after  = ( ! empty( $item['keyword_after'] ) ? $this->fix_for_apostophie( $item['keyword_after'] ) : '' );
			$left_boundary  = isset( $item['left_boundary'] ) ? $this->get_boundary( $item['left_boundary'] ) : '';
			$right_boundary = isset( $item['right_boundary'] ) ? $this->get_boundary( $item['right_boundary'] ) : '';
			$limit          = (int) ( ! empty( $item['limit'] ) ? $item['limit'] : 100 );

			// Check if 'none' option is selected (match anywhere, no boundaries)
			$is_none_boundary = ( isset( $item['left_boundary'] ) && $item['left_boundary'] === 'none' ) || 
			                     ( isset( $item['right_boundary'] ) && $item['right_boundary'] === 'none' );
			
			// Check if white space boundary is selected (empty string means whitespace boundary)
			$is_whitespace_boundary = ( isset( $item['left_boundary'] ) && $item['left_boundary'] === '' ) || 
			                          ( isset( $item['right_boundary'] ) && $item['right_boundary'] === '' );

			// For 'none' and whitespace boundaries, ignore keyword_before and keyword_after
			if ( $is_none_boundary || $is_whitespace_boundary ) {
				$keyword_before = '';
				$keyword_after = '';
			}

			// Determine word boundaries based on the boundary type
			$boundary_start = '';
			$boundary_end = '';
			
			if ( ! $is_none_boundary ) {
				// Use word boundaries unless 'none' is selected
				$boundary_start = '\b';
				$boundary_end = '\b';
			}

			// step 1: added placeholder.
			$content = preg_replace_callback(
				'/' . $boundary_start . '(' . $keyword_before . ')(' . $left_boundary . ')(' . $tags . ')(' . $right_boundary . ')(' . $keyword_after . ')' . $boundary_end . '/' . $search_mode,
				function ( $match ) use ( $btl_plc_space, $autolink_url, $attribute, $btl_plc_svg ) {
					return $match[1] . $match[2] .
					'<a ' .
					$btl_plc_space .
					'class="' . $this->empty_placeholder . 'btl_autolink_hyperlink" ' .
					$btl_plc_space .
					'href="' . $autolink_url . '" ' .
					$btl_plc_space .
					$attribute . '>' .
					$btl_plc_svg .
					$match[3] . $this->empty_placeholder .
					'</a>' .
					$match[4] . $match[5];
				},
				$content,
				$limit
			);

			// Protect newly created anchor tags from being matched by subsequent keywords
			// This prevents keywords from matching inside the anchor tags we just created
			$content = preg_replace_callback(
				'/<a[^>]*?>[\s\S]*?<\/a>/u',
				array( $this, 'replace_protected_tags_and_contents_by_placeholder' ),
				$content
			);
		}

		// Restore protected placeholders after keyword matching
		$content = str_replace( array_values( $placeholder_protection_map ), array_keys( $placeholder_protection_map ), $content );

		$content            = $this->put_back_protected_tags_contents( $content );
		$content            = $this->put_back_all_tag_attributes( $content );
		
		// Generate icon SVG - use custom icon if available, otherwise use default
		$hyperlink_icon_svg = '';
		if ( $this->is_show_icon ) {
			if ( !empty( $this->custom_icon_svg ) ) {
				// Use custom uploaded SVG icon (sanitized with dynamic sizing)
				$hyperlink_icon_svg = ' ' . $this->sanitize_custom_svg( $this->custom_icon_svg, $this->keyword_styles['fontSize'] );
			} else {
				// Use default SVG icon
				$hyperlink_icon_svg = ' <svg class="btl_autolink_icon_svg" enable-background="new 0 0 64 64"  viewBox="0 0 64 64"  xmlns="http://www.w3.org/2000/svg"><g><g ><g><path d="m36.243 29.758c-.16 0-1.024-.195-1.414-.586-3.119-3.119-8.194-3.12-11.314 0-.78.781-2.048.781-2.828 0-.781-.781-.781-2.047 0-2.828 4.679-4.68 12.292-4.679 16.97 0 .781.781.781 2.047 0 2.828-.39.391-.903.586-1.414.586z"/></g></g><g ><g><path d="m34.829 41.167c-3.073 0-6.146-1.17-8.485-3.509-.781-.781-.781-2.047 0-2.828.78-.781 2.048-.781 2.828 0 3.119 3.119 8.194 3.12 11.314 0 .78-.781 2.048-.781 2.828 0 .781.781.781 2.047 0 2.828-2.34 2.339-5.413 3.509-8.485 3.509z"/></g></g><g ><g><path d="m41.899 38.243c-.16 0-1.024-.195-1.414-.586-.781-.781-.781-2.047 0-2.828l11.172-11.172c.78-.781 2.048-.781 2.828 0 .781.781.781 2.047 0 2.828l-11.172 11.172c-.39.391-.902.586-1.414.586z"/></g></g><g ><g><path d="m25.071 55.071c-.16 0-1.024-.195-1.414-.586-.781-.781-.781-2.047 0-2.828l6.245-6.245c.78-.781 2.048-.781 2.828 0 .781.781.781 2.047 0 2.828l-6.245 6.245c-.39.391-.902.586-1.414.586z"/></g></g><g ><g><path d="m10.929 40.929c-.16 0-1.024-.195-1.414-.586-.781-.781-.781-2.047 0-2.828l11.172-11.171c.781-.781 2.048-.781 2.828 0 .781.781.781 2.047 0 2.828l-11.172 11.171c-.391.39-.903.586-1.414.586z"/></g></g><g ><g><path d="m32.684 19.175c-.16 0-1.023-.195-1.414-.585-.781-.781-.781-2.047 0-2.829l6.245-6.246c.781-.781 2.047-.781 2.829 0 .781.781.781 2.047 0 2.829l-6.245 6.246c-.391.389-.904.585-1.415.585z"/></g></g><g ><g><path d="m18 57.935c-3.093 0-6.186-1.15-8.485-3.45-4.6-4.6-4.6-12.371 0-16.971.78-.781 2.048-.781 2.828 0 .781.781.781 2.047 0 2.828-3.066 3.066-3.066 8.248 0 11.314s8.248 3.066 11.314 0c.78-.781 2.048-.781 2.828 0 .781.781.781 2.047 0 2.828-2.299 2.301-5.392 3.451-8.485 3.451z"/></g></g><g ><g><path d="m53.071 27.071c-.16 0-1.024-.195-1.414-.586-.781-.781-.781-2.047 0-2.828 3.066-3.066 3.066-8.248 0-11.314s-8.248-3.066-11.314 0c-.78.781-2.048.781-2.828 0-.781-.781-.781-2.047 0-2.828 4.6-4.6 12.371-4.6 16.971 0s4.6 12.371 0 16.971c-.391.39-.903.585-1.415.585z"/></g></g></g></svg>';
			}
		}
		// step 3: remove unnecessary strings.
		$content = str_replace(
			array(
				$this->empty_placeholder,
				$btl_plc_space,
				$btl_plc_colon,
				$btl_plc_slash,
				$btl_plc_svg,
			),
			array(
				'',
				' ',
				':',
				'/',
				$hyperlink_icon_svg,
			),
			$content
		);
		return $content;
	}

	public function get_keywords() {
		$keywords = \BetterLinks\Helper::get_keywords();
		$keywords = $this->prepare_keywords( $keywords );
		return $keywords;
	}

	public function prepare_keywords( $keywords ) {
		$single_keywords          = array();
		$comma_separated_keywords = array();
		if ( is_array( $keywords ) ) {
			foreach ( $keywords as &$value ) {
				$temp             = json_decode( $value, true );
				$tags             = $this->keywords_to_tags_generator( $temp['keywords'] );
				$temp['keywords'] = $tags;
				$value            = $temp;
				unset($temp, $tags);
				
				if ( strpos( $value['keywords'], '|' ) === false ) {
					array_push( $single_keywords, $value );
				} else {
					array_push( $comma_separated_keywords, $value );
				}
			}
		}

		$keywords = $this->get_sorted_keywords_high_to_low_length( $comma_separated_keywords, $single_keywords );
		unset( $comma_separated_keywords, $single_keywords );
		return $keywords;
	}

	/**
	 * Returns the sorted keywords high to low lengths order
	 *
	 * @param array $single_keywords - Keywords array without '|'.
	 * @param array $comma_separated_keywords - Keywords array with '|', basically these are the comma separated keywords
	 *
	 * @return array $keywords - Combined $single_keywords & array2 keywords in high to low length order.
	 */
	public function get_sorted_keywords_high_to_low_length( $comma_separated_keywords, $single_keywords  ) {
		// 👇 Sorting $keywords by their length - high to low.
		uasort(
			$single_keywords,
			function ( $a, $b ) {
				return strlen( $b['keywords'] ) - strlen( $a['keywords'] );
			}
		);
		uasort(
			$comma_separated_keywords,
			function ( $a, $b ) {
				return strlen( $b['keywords'] ) - strlen( $a['keywords'] );
			}
		);
		return array_merge( $comma_separated_keywords, $single_keywords );
	}

	public function keywords_to_tags_generator( $string ) {
		$string = trim( $string );
		// Split by comma to get individual keywords
		$keywords = preg_split( '/\,\s+|,+/', $string );
		// Sort keywords by length (longest first) to prevent shorter keywords from matching inside longer ones
		usort( $keywords, function( $a, $b ) {
			return strlen( $b ) - strlen( $a );
		});
		// Rejoin with | for regex alternation
		$string = implode( '|', $keywords );
		return $string;
	}
	public function get_boundary( $data ) {
		$boundary = '';
		switch ( $data ) {
			case 'generic':
				$boundary = '\b';
				break;

			case 'test':
				// White Space - match only when keyword has whitespace before AND after
				$boundary = '\s';
				break;

			case 'whitespace':
				$boundary = '\b \b';
				break;

			case 'comma':
				$boundary = ',';
				break;

			case 'point':
				$boundary = '\.';
				break;

			case 'none':
				// None - match anywhere, no boundary check
				$boundary = '';
				break;
		}
		return $boundary;
	}
	public function get_link_attributes( $item ) {
		// $this->empty_placeholder added to make it unique string so that, strings like 'target','rel','nofollow' don't get autolinked/hyperlinked
		$attribute = ' ';
		
		// Use individual item settings if available, otherwise fallback to global settings
		if ( isset( $this->global_alk_settings['openNewTab'] ) && $this->global_alk_settings['openNewTab'] ) {
			$attribute .= ' ' . $this->empty_placeholder . 'target="' . $this->empty_placeholder . '_blank"';
		}
		
		return $attribute;
	}
	public function prevent_protected_tags_contents_from_getting_autolinked( $content ) {
		global $betterlinks;
		$autolink_in_heading_regex_part = isset( $betterlinks['is_autolink_headings'] ) && ! $betterlinks['is_autolink_headings']
			? '|<h[1-6][^>]*?>[\s\S]*?<\/h[1-6]>'
			: '';
		$content                        = preg_replace_callback(
			'/<a[^>]*?>[\s\S]*?<\/a>' .
			'|<img[^>]*?>' .
			'|<script[^>]*?>[\s\S]*?<\/script>' .
			$autolink_in_heading_regex_part .
			'/u',
			array( $this, 'replace_protected_tags_and_contents_by_placeholder' ),
			$content
		);
		return $content;
	}
	public function prevent_all_tag_attributes_from_getting_autolinked( $content ) {
		$content = preg_replace_callback(
			'/<[^>]*?>/u',
			array( $this, 'replace_protected_tags_and_attribute_only_by_placeholder' ),
			$content
		);
		return $content;
	}

	public function replace_protected_tags_and_contents_by_placeholder( $match ) {
		$position = count( $this->protected_tags_content_lists );
		array_push( $this->protected_tags_content_lists, $match[0] );
		return '[alkpt]' . $position . '[/alkpt]';
	}
	public function replace_protected_tags_and_attribute_only_by_placeholder( $match ) {
		$position = count( $this->protected_tags_content_lists );
		array_push( $this->protected_tags_content_lists, $match[0] );
		return '[alkpta]' . $position . '[/alkpta]';
	}
	public function put_back_protected_tags_contents( $content ) {
		$content = preg_replace_callback(
			'/\[alkpt\](\d+)\[\/alkpt\]/u',
			array( $this, 'replace_placeholders_with_the_stored_tags_contents' ),
			$content
		);
		return $content;
	}
	public function put_back_all_tag_attributes( $content ) {
		$content = preg_replace_callback(
			'/\[alkpta\](\d+)\[\/alkpta\]/u',
			array( $this, 'replace_placeholders_with_the_stored_tag_attributes' ),
			$content
		);
		return $content;
	}
	public function replace_placeholders_with_the_stored_tags_contents( $match ) {
		return $this->protected_tags_content_lists[ $match[1] ];
	}
	public function replace_placeholders_with_the_stored_tag_attributes( $match ) {
		return $this->protected_tags_content_lists[ $match[1] ];
	}
	public function autolink_css() {
		// If custom styles are not enabled, use minimal CSS to inherit theme styles
		if ( ! $this->enable_custom_styles ) {
			?>
			<style>
				/* BetterLinks Auto-Link Keywords - Inherit Theme Styling */
				a.btl_autolink_hyperlink {
					/* All styles will inherit from theme by default */
				}
			</style>
			<?php
			return;
		}
		
		// Custom styles are enabled - apply user-defined styles
		// Use background color directly - it can be hex, rgb, or rgba format
		$bg_color = $this->keyword_styles['backgroundColor'];
		
		?>
		<style>
			/* BetterLinks Auto-Link Keywords Custom Styling */
			a.btl_autolink_hyperlink {
				<?php if ( $this->keyword_styles['textColor'] !== 'inherit' ) : ?>
				color: <?php echo esc_attr( $this->keyword_styles['textColor'] ); ?> !important;
				<?php endif; ?>
				<?php if ( $this->keyword_styles['fontWeight'] !== 'inherit' ) : ?>
				font-weight: <?php echo esc_attr( $this->keyword_styles['fontWeight'] ); ?> !important;
				<?php endif; ?>
				<?php if ( $this->keyword_styles['fontStyle'] !== 'inherit' ) : ?>
				font-style: <?php echo esc_attr( $this->keyword_styles['fontStyle'] ); ?> !important;
				<?php endif; ?>
				<?php if ( $this->keyword_styles['fontSize'] !== 'inherit' && $this->keyword_styles['fontSize'] !== 'unset' ) : ?>
				font-size: <?php echo esc_attr( $this->keyword_styles['fontSize'] ); ?>px !important;
				<?php endif; ?>
				<?php if ( $this->keyword_styles['textDecoration'] !== 'inherit' ) : ?>
				text-decoration: <?php echo esc_attr( $this->keyword_styles['textDecoration'] ); ?> !important;
				<?php endif; ?>
				<?php if ( !empty( $bg_color ) && $bg_color !== 'transparent' && $bg_color !== '#ffffff' && $bg_color !== 'rgba(255, 255, 255, 0)' && $bg_color !== 'rgba(255,255,255,0)' ) : ?>
				background-color: <?php echo esc_attr( $bg_color ); ?> !important;
				<?php 
				// Dynamic padding based on font size for background
				$fontSize = $this->keyword_styles['fontSize'] !== 'inherit' && $this->keyword_styles['fontSize'] !== 'unset' ? intval( $this->keyword_styles['fontSize'] ) : 14;
				$basePadding = max( $fontSize * 0.15, 2 );
				?>
				padding: <?php echo esc_attr( $basePadding ); ?>px <?php echo esc_attr( $basePadding ); ?>px !important;
				border-radius: 3px !important;
				<?php endif; ?>
				transition: all 0.2s ease !important;
			}
			
			<?php if ( $this->is_show_icon && !empty( $bg_color ) && $bg_color !== 'transparent' && $bg_color !== '#ffffff' && $bg_color !== 'rgba(255, 255, 255, 0)' && $bg_color !== 'rgba(255,255,255,0)' ) : ?>
			/* Special padding for keywords with both background and icon */
			a.btl_autolink_hyperlink {
				<?php 
				// Dynamic padding for background with icon: scale left padding with font size
				$fontSize = $this->keyword_styles['fontSize'] !== 'inherit' && $this->keyword_styles['fontSize'] !== 'unset' ? intval( $this->keyword_styles['fontSize'] ) : 14;
				$basePadding = max( $fontSize * 0.15, 2 );
				$leftPadding = max( $fontSize + 8, 20 ); // Minimum 20px, scales with font size
				?>
				padding: <?php echo esc_attr( $basePadding ); ?>px <?php echo esc_attr( $basePadding ); ?>px <?php echo esc_attr( $basePadding ); ?>px <?php echo esc_attr( $leftPadding ); ?>px !important;
			}
			<?php endif; ?>
			
			/* Hover effect for better user experience */
			<?php if ( $this->keyword_styles['hoverTextColor'] !== 'inherit' ) : ?>
			a.btl_autolink_hyperlink:hover {
				color: <?php echo esc_attr( $this->keyword_styles['hoverTextColor'] ); ?> !important;
				<?php if ( $this->keyword_styles['textDecoration'] === 'none' ) : ?>
				text-decoration: underline !important;
				<?php endif; ?>
			}
			<?php endif; ?>
			
			/* Focus state for accessibility */
			
		</style>
		<?php
	}
	
	public function autolink_icon_css() {
		// Get font size - use inherit value if custom styles are not enabled
		$fontSize = 14; // Default fallback
		if ( $this->enable_custom_styles && $this->keyword_styles['fontSize'] !== 'inherit' && $this->keyword_styles['fontSize'] !== 'unset' ) {
			$fontSize = intval( $this->keyword_styles['fontSize'] );
		}
		//echo "padding: $leftPadding px !important;"; die();
		?>
		<style>
			/* BetterLinks Auto-Link Keywords Icon Styling */
			a.btl_autolink_hyperlink {
				position: relative !important;
				<?php 
				// Dynamic padding based on font size for icon links
				$leftPadding = max( $fontSize + 8, 20 ); // Minimum 20px, scales with font size
				$padding_min = $leftPadding > 29 ? $leftPadding : max( $leftPadding - 5, 0 );
				?>
				padding: 0 5px 0 <?php echo esc_attr( $leftPadding > 40 ? $leftPadding + 5 : $padding_min ); ?>px !important;
				display: inline-block;
			}

			svg.btl_autolink_icon_svg, .btl_autolink_icon_svg {
				width: <?php echo esc_attr( $fontSize > 0 ? $fontSize - 4 : 14 ); ?>px !important;
				height: <?php echo esc_attr( $fontSize > 0 ? $fontSize - 4 : 14 ); ?>px !important;
				<?php 
				// Dynamic left position based on font size
				$leftPosition = max( $fontSize * 0.3, 2 );
				?>
				left: <?php echo esc_attr( $leftPosition ); ?>px !important;
				top: 50% !important;
				transform: translateY(-50%) !important;
				position: absolute !important;
				/* fill: currentColor !important; */
			}
		</style>
		<?php
	}
	public function make_url_string_comparable( $url_string = '' ) {
		return rtrim( strtolower( preg_replace( '/https?:\/\//i', '', $url_string ) ), '/' );
	}
	public function fix_for_apostophie( $tags = '' ) {
		return preg_replace( "/\’|\‘|\'|\&\#8217\;|\&\#8219\;/", "(?:[\'\’\‘]|\&\#8217\;|\&\#8219\;)", $tags );
	}
}
