<?php

namespace BitApps\BTCBI_PRO\Triggers\EasyCommerce;

use BitCode\FI\Core\Util\Helper;
use BitCode\FI\Flow\Flow;

final class EasyCommerceController
{
    public static function info()
    {
        return [
            'name'              => 'EasyCommerce',
            'type'              => 'custom_form_submission',
            'is_active'         => EasyCommerceHelper::isPluginInstalled(),
            'documentation_url' => 'https://bit-integrations.com/wp-docs/trigger/easycommerce-integrations/',
            'tutorial_url'      => '#',
            'tasks'             => [
                'action' => 'easy_commerce/get',
                'method' => 'get',
            ],
            'fetch' => [
                'action' => 'trigger/test',
                'method' => 'post',
            ],
            'fetch_remove' => [
                'action' => 'trigger/test/remove',
                'method' => 'post',
            ],
            'isPro' => true
        ];
    }

    public function getAllTasks()
    {
        if (!EasyCommerceHelper::isPluginInstalled()) {
            wp_send_json_error(\sprintf(__('%s is not installed or activated', 'bit-integrations-pro'), 'EasyCommerce'));
        }

        wp_send_json_success(StaticData::forms());
    }

    public static function handleUserCreated($userId, $user)
    {
        $formData = Helper::prepareFetchFormatFields((array) $user);

        return self::flowExecute('easycommerce_user_created', $formData);
    }

    public static function handleUserUpdated($userId, $user)
    {
        $formData = Helper::prepareFetchFormatFields((array) $user);

        return self::flowExecute('easycommerce_user_updated', $formData);
    }

    public static function handleAttributeCreated($attributeId)
    {
        $formData = EasyCommerceHelper::FormatAttributeData($attributeId);

        return self::flowExecute('easycommerce_attribute_added', $formData);
    }

    public static function handleAddToCart($productData)
    {
        if (!empty($productData[0])) {
            $productData = reset($productData);
        }

        if (empty($productData['id'])) {
            return;
        }

        $formData = EasyCommerceHelper::FormatProductData($productData['id'], false, $productData['quantity'] ?? false);

        return self::flowExecute('easycommerce_add_to_cart', $formData);
    }

    public static function handleCartUpdated($productId, $priceId, $quantity)
    {
        $formData = EasyCommerceHelper::FormatProductData($productId, false, $quantity);

        return self::flowExecute('easycommerce_update_cart', $formData);
    }

    public static function handleRemoveFromCart($productId)
    {
        $formData = EasyCommerceHelper::FormatProductData($productId);

        return self::flowExecute('easycommerce_remove_from_cart', $formData);
    }

    public static function handleClearedCard($request)
    {
        $request = EasyCommerceHelper::getRequestData($request);
        $formData = Helper::prepareFetchFormatFields($request);

        return self::flowExecute('easycommerce_clear_cart', $formData);
    }

    public static function handleShippingMethod($id)
    {
        $formData = EasyCommerceHelper::FormatShippingMethodData($id);

        return self::flowExecute('easycommerce_set_shipping_method', $formData);
    }

    public static function handleCouponApplied($coupon)
    {
        if (empty($coupon)) {
            return;
        }

        $data = EasyCommerceHelper::getCouponData($coupon);

        $formData = Helper::prepareFetchFormatFields($data);

        return self::flowExecute('easycommerce_apply_coupon', $formData);
    }

    public static function handleCouponRemoved($code, $request)
    {
        if (empty($code) || !class_exists('\EasyCommerce\Models\Coupon')) {
            return;
        }

        $coupon = new Coupon($code);

        $request = EasyCommerceHelper::getRequestData($request);
        $request['coupon'] = EasyCommerceHelper::getCouponData($coupon);

        $formData = Helper::prepareFetchFormatFields($request);

        return self::flowExecute('easycommerce_remove_coupon', $formData);
    }

    public static function handleCouponDeleted($id)
    {
        $formData = EasyCommerceHelper::FormatCouponData($id);

        return self::flowExecute('easycommerce_before_delete_coupon', $formData);
    }

    public static function handleCouponStatusUpdated($id, $status)
    {
        $formData = EasyCommerceHelper::FormatCouponData($id, $status);

        return self::flowExecute('easycommerce_coupon_status_updated', $formData);
    }

    public static function handleOrderCreated($orderId, $params)
    {
        $formData = EasyCommerceHelper::FormatOrderData($orderId, $params);

        return self::flowExecute('easycommerce_after_create_order', $formData);
    }

    public static function handleOrderStatusUpdated($orderId, $newStatus, $oldStatus)
    {
        return self::handleOrderStatusRelatedTrigger(
            $orderId,
            $newStatus,
            $oldStatus,
            'easycommerce_order_status_updated'
        );
    }

    public static function handleOrderStatusSetToProcessing($orderId, $newStatus, $oldStatus)
    {
        if ($newStatus !== 'processing') {
            return;
        }

        return self::handleOrderStatusRelatedTrigger(
            $orderId,
            $newStatus,
            $oldStatus,
            'easycommerce_set_order_status_to_processing'
        );
    }

    public static function handleOrderStatusSetToCompleted($orderId, $newStatus, $oldStatus)
    {
        if ($newStatus !== 'completed') {
            return;
        }

        return self::handleOrderStatusRelatedTrigger(
            $orderId,
            $newStatus,
            $oldStatus,
            'easycommerce_set_order_status_to_completed'
        );
    }

    public static function handleOrderStatusSetToPending($orderId, $newStatus, $oldStatus)
    {
        if ($newStatus !== 'pending') {
            return;
        }

        return self::handleOrderStatusRelatedTrigger(
            $orderId,
            $newStatus,
            $oldStatus,
            'easycommerce_set_order_status_to_pending'
        );
    }

    public static function handleOrderStatusSetToRefunded($orderId, $newStatus, $oldStatus)
    {
        if ($newStatus !== 'refunded') {
            return;
        }

        return self::handleOrderStatusRelatedTrigger(
            $orderId,
            $newStatus,
            $oldStatus,
            'easycommerce_set_order_status_to_refunded'
        );
    }

    public static function handleOrderStatusSetToOnHold($orderId, $newStatus, $oldStatus)
    {
        if ($newStatus !== 'on_hold') {
            return;
        }

        return self::handleOrderStatusRelatedTrigger(
            $orderId,
            $newStatus,
            $oldStatus,
            'easycommerce_set_order_status_to_on_hold'
        );
    }

    public static function handleOrderStatusSetToCancelled($orderId, $newStatus, $oldStatus)
    {
        if ($newStatus !== 'cancelled') {
            return;
        }

        return self::handleOrderStatusRelatedTrigger(
            $orderId,
            $newStatus,
            $oldStatus,
            'easycommerce_set_order_status_to_cancelled'
        );
    }

    public static function handleOrderRefunded($order, $reason)
    {
        $orderData = EasyCommerceHelper::getOrderData($order);

        $orderData['refund_reason'] = $reason;

        $formData = Helper::prepareFetchFormatFields($orderData);

        return self::flowExecute('easycommerce_after_refund_order', $formData);
    }

    public static function handleEmailCustomer($email, $subject, $body, $placeholders)
    {
        $data = [
            'email'        => $email,
            'subject'      => $subject,
            'body'         => $body,
            'placeholders' => \is_array($placeholders) ? json_encode($placeholders) : $placeholders,
        ];

        $formData = Helper::prepareFetchFormatFields($data);

        return self::flowExecute('easycommerce_email', $formData);
    }

    public static function handleProductCreated($productData)
    {
        if (empty($productData['id'])) {
            return;
        }

        $formData = EasyCommerceHelper::FormatProductData($productData['id']);

        return self::flowExecute('easycommerce_create_product', $formData);
    }

    public static function handleProductUpdated($productData)
    {
        if (empty($productData['id'])) {
            return;
        }

        $formData = EasyCommerceHelper::FormatProductData($productData['id']);

        return self::flowExecute('easycommerce_update_product', $formData);
    }

    public static function handleProductDeleted($id)
    {
        $formData = EasyCommerceHelper::FormatProductData($id);

        return self::flowExecute('easycommerce_before_delete_product', $formData);
    }

    public static function handleProductStatusUpdated($status, $id)
    {
        return self::handleProductStatusRelatedTrigger(
            $status,
            $id,
            'easycommerce_product_status_updated_bulk'
        );
    }

    public static function handleProductStatusUpdatedToLive($status, $id)
    {
        if ($status !== 'publish') {
            return;
        }

        return self::handleProductStatusRelatedTrigger(
            $status,
            $id,
            'easycommerce_product_status_updated_to_live_bulk'
        );
    }

    public static function handleProductStatusUpdatedToDraft($status, $id)
    {
        if ($status !== 'draft') {
            return;
        }

        return self::handleProductStatusRelatedTrigger(
            $status,
            $id,
            'easycommerce_product_status_updated_to_draft_bulk'
        );
    }

    public static function handleProductStatusUpdatedToTrash($status, $id)
    {
        if ($status !== 'trash') {
            return;
        }

        return self::handleProductStatusRelatedTrigger(
            $status,
            $id,
            'easycommerce_product_status_updated_to_trash_bulk'
        );
    }

    public static function handleSendAbandonedReminder($hash, $request)
    {
        $formData = EasyCommerceHelper::formatAbandonedCartData($hash, $request);

        return self::flowExecute('easycommerce_before_send_abandoned_reminder', $formData);
    }

    public static function handleAbandonedCartDeleted($hash, $isDeleted, $request)
    {
        $formData = EasyCommerceHelper::formatAbandonedCartData($hash, $request, true, $isDeleted);

        return self::flowExecute('easycommerce_after_remove_abandoned', $formData);
    }

    private static function handleProductStatusRelatedTrigger($status, $id, $entityId)
    {
        $formData = EasyCommerceHelper::FormatProductData($id, $status);

        return self::flowExecute($entityId, $formData);
    }

    private static function handleOrderStatusRelatedTrigger($orderId, $newStatus, $oldStatus, $entityId)
    {
        $formData = EasyCommerceHelper::FormatOrderData($orderId, [], $newStatus, $oldStatus);

        return self::flowExecute($entityId, $formData);
    }

    private static function flowExecute($entityId, $formData)
    {
        if (empty($formData) || !\is_array($formData)) {
            return;
        }

        Helper::setTestData("btcbi_{$entityId}_test", array_values($formData));

        $flows = Flow::exists('EasyCommerce', $entityId);

        if (!$flows) {
            return;
        }

        $data = array_column($formData, 'value', 'name');

        Flow::execute('EasyCommerce', $entityId, $data, $flows);

        return ['type' => 'success'];
    }
}
